package com.bxm.thirdparty.platform.queue.impl;

import cn.hutool.core.date.DateField;
import cn.hutool.core.date.DateTime;
import cn.hutool.core.date.DateUtil;
import com.bxm.newidea.component.JSON;
import com.bxm.newidea.component.uuid.SequenceCreater;
import com.bxm.thirdparty.platform.config.ServerNotifyProperties;
import com.bxm.thirdparty.platform.mapper.BusinessNotifyLogMapper;
import com.bxm.thirdparty.platform.model.entity.BusinessNotifyLogEntity;
import com.bxm.thirdparty.platform.queue.NotifyQueueCacheHolder;
import com.bxm.thirdparty.platform.queue.NotifyQueueExecutor;
import com.bxm.thirdparty.platform.queue.QueueService;
import com.bxm.thirdparty.platform.queue.bo.QueueBO;
import com.bxm.thirdparty.platform.queue.notifybusiness.enums.NotifyStatusEnum;
import com.bxm.thirdparty.platform.service.ApplicationRelationService;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import java.util.Date;
import java.util.Objects;

/**
 * @author lowi
 * @date 2023/4/6 15:38
 */
@Service
@Slf4j
@AllArgsConstructor
public class QueueServiceImpl implements QueueService {

    private final BusinessNotifyLogMapper businessNotifyLogMapper;

    private final ApplicationRelationService applicationRelationService;

    private final SequenceCreater sequenceCreater;

    private final ServerNotifyProperties properties;

    private final NotifyQueueCacheHolder queueCacheHolder;

    private final NotifyQueueExecutor notifyQueueExecutor;


    @Override
    public void submitBusinessNotify(QueueBO queueBO) {

        if (queueBO.getNotifyUrl() == null) {
            String notifyUrl = applicationRelationService.getNotifyUrl(queueBO.getApplicationName());
            queueBO.setNotifyUrl(notifyUrl);
        }

        Date nextNotifyTime;
        Long notifyId = sequenceCreater.nextLongId();
        int status = NotifyStatusEnum.WAIT_ING.getStatus();
        if (queueBO.getDelayTime() != null) {
            DateTime offset = DateUtil.offset(new Date(), DateField.SECOND, queueBO.getDelayTime());
            nextNotifyTime = offset.toJdkDate();
        } else {
            nextNotifyTime = new Date();
            //直接通知
            if (notifyQueueExecutor.isFullLoad()) {
                queueCacheHolder.addToQueue(queueBO);
            } else {
                notifyQueueExecutor.executor(queueBO);
            }
            status = NotifyStatusEnum.QUEUE_ING.getStatus();
        }
        BusinessNotifyLogEntity businessNotifyLogEntity = new BusinessNotifyLogEntity();
        businessNotifyLogEntity.setId(notifyId);
        businessNotifyLogEntity.setRequestId(queueBO.getRequestId());
        businessNotifyLogEntity.setRequest(JSON.toJSONString(queueBO));
        businessNotifyLogEntity.setFailNum(0);
        businessNotifyLogEntity.setCreateTime(new Date());
        businessNotifyLogEntity.setNextNotifyTime(nextNotifyTime);
        businessNotifyLogEntity.setStatus(status);
        businessNotifyLogEntity.setType(queueBO.getRequestBodyBO().getType());
        businessNotifyLogMapper.insert(businessNotifyLogEntity);
    }


    @Override
    public void addBusinessFailInfo(String requestId, String request, String errorMsg) {
        BusinessNotifyLogEntity businessNotifyLogEntity = businessNotifyLogMapper.getByRequestId(requestId);
        if (Objects.isNull(businessNotifyLogEntity)) {
            int nextInterval = properties.getRetryIntervalArray()[1];
            DateTime offset = DateUtil.offset(new Date(), DateField.SECOND, nextInterval);
            businessNotifyLogEntity = new BusinessNotifyLogEntity();
            businessNotifyLogEntity.setId(sequenceCreater.nextLongId());
            businessNotifyLogEntity.setRequest(request);
            businessNotifyLogEntity.setRequestId(requestId);
            businessNotifyLogEntity.setFailNum(0);
            businessNotifyLogEntity.setCreateTime(new Date());
            businessNotifyLogEntity.setNextNotifyTime(offset.toJdkDate());
            businessNotifyLogEntity.setStatus(NotifyStatusEnum.WAIT_ING.getStatus());
            businessNotifyLogMapper.insert(businessNotifyLogEntity);
        } else {
            if (businessNotifyLogEntity.getFailNum() + 1 >= properties.getRetryIntervalArray().length) {
                log.error(
                        "业务通知[{}]重试次数达到[{}]次，直接记录为失败,最近一次失败信息为：{}",
                        businessNotifyLogEntity.getId(),
                        businessNotifyLogEntity.getFailNum(),
                        errorMsg
                );
                return;
            }
            Long id = businessNotifyLogEntity.getId();
            Integer failNum = businessNotifyLogEntity.getFailNum() + 1;
            int nextInterval = properties.getRetryIntervalArray()[businessNotifyLogEntity.getFailNum()];
            DateTime offset = DateUtil.offset(new Date(), DateField.SECOND, nextInterval);

            businessNotifyLogEntity = new BusinessNotifyLogEntity();
            businessNotifyLogEntity.setNextNotifyTime(offset.toJdkDate());
            businessNotifyLogEntity.setId(id);
            //失败了，更新状态成的待处理
            businessNotifyLogEntity.setStatus(NotifyStatusEnum.WAIT_ING.getStatus());
            businessNotifyLogEntity.setFailNum(failNum);
            businessNotifyLogEntity.setModifyTime(new Date());
            businessNotifyLogEntity.setRemark(errorMsg);
            businessNotifyLogMapper.updateById(businessNotifyLogEntity);
        }
    }

    @Override
    public void updateNotifyInfoSuccess(String requestId) {
        BusinessNotifyLogEntity businessNotifyLogEntity = businessNotifyLogMapper.getByRequestId(requestId);
        if (Objects.isNull(businessNotifyLogEntity)) {
            log.error(
                    "业务通知[{}]成功，但未查询到通知信息",
                    requestId
            );
            return;
        }
        BusinessNotifyLogEntity updateClass = new BusinessNotifyLogEntity();
        updateClass.setId(businessNotifyLogEntity.getId());
        updateClass.setStatus(NotifyStatusEnum.SUCCESS.getStatus());
        updateClass.setModifyTime(new Date());
        businessNotifyLogMapper.updateById(updateClass);
    }

}
