package com.bxm.thirdparty.platform.adapter.chargephone.xinrong;

import com.bxm.newidea.component.JSON;
import com.bxm.newidea.component.bo.Message;
import com.bxm.newidea.component.tools.StringUtils;
import com.bxm.thirdparty.platform.adapter.chargephone.ChargePhoneAction;
import com.bxm.thirdparty.platform.adapter.chargephone.xinrong.bo.XinRongChargeBO;
import com.bxm.thirdparty.platform.adapter.chargephone.xinrong.bo.XinRongCreateOrderBO;
import com.bxm.thirdparty.platform.adapter.chargephone.xinrong.enums.XinRongChargeCouponEnum;
import com.bxm.thirdparty.platform.adapter.chargephone.xinrong.utils.PhoneUtil;
import com.bxm.thirdparty.platform.adapter.chargephone.xinrong.utils.XinRongUtil;
import com.bxm.thirdparty.platform.adapter.context.PlatformContext;
import com.bxm.thirdparty.platform.config.ThirdPartyConfigProperties;
import com.bxm.thirdparty.platform.constant.CommonConstant;
import com.bxm.thirdparty.platform.context.ThreadContext;
import com.bxm.thirdparty.platform.enums.PlatformEnum;
import com.bxm.thirdparty.platform.facade.request.ChargePhoneRequest;
import com.bxm.thirdparty.platform.facade.response.ChargePhoneTopUpResponse;
import com.bxm.thirdparty.platform.model.entity.ChargesPhoneLogEntity;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.math.BigDecimal;
import java.util.Objects;

/**
 * @author lowi
 * @date 2023/4/6 19:07
 */
@Component
@Slf4j
public class XinRongChargePhoneAction implements ChargePhoneAction<ChargePhoneRequest, ChargesPhoneLogEntity> {

    @Resource
    private ThirdPartyConfigProperties thirdPartyConfigProperties;

    @Override
    public PlatformEnum support() {
        return PlatformEnum.XIN_RONG;
    }

    @Override
    public Message exec(PlatformContext<ChargePhoneRequest, ChargesPhoneLogEntity> context) {
        ChargePhoneRequest request = context.getRequest();
        ChargesPhoneLogEntity orderInfo = context.getOrderInfo();

        String phoneNo = request.getPhoneNo();
        String phoneCarrier = getPhoneCarrier(phoneNo);
        BigDecimal money = request.getMoney();
        XinRongChargeCouponEnum chargeCouponEnum = XinRongChargeCouponEnum.get(phoneCarrier, money);
        if (Objects.isNull(chargeCouponEnum)) {
            log.error("没有匹配的充值金额，参数错误，请检查逻辑，金额为：{}", money);
            return Message.build(false).setMessage("充值金额不存在");
        }
        XinRongUtil xinRongUtil = new XinRongUtil(thirdPartyConfigProperties);
        //充值前先判断余额充足不充足
        if (xinRongUtil.getAccountBalance().compareTo(money) < 0) {
            log.error("心融话费充值失败，充值余额不足：{}", request);
            return Message.build(false).setMessage("充值余额不足");
        }

        ChargePhoneTopUpResponse response = new ChargePhoneTopUpResponse();

        String body = xinRongUtil.topUp(request, orderInfo.getOrderNo().toString(),
                chargeCouponEnum);
        if (Objects.isNull(body)) {
            return Message.build(false).setBizCode(CommonConstant.THIRD_PARTY_500).setMessage("调用第三方话费充值失败");
        }

        XinRongChargeBO xinRongChargeBO = JSON.parseObject(body, XinRongChargeBO.class);
        if (Objects.isNull(xinRongChargeBO) || !Objects.equals(xinRongChargeBO.getCode(), XinRongUtil.SUCCESS_CODE)) {
            log.error("调取第三方话费充值失败：param:{},result:{}", context, body);
            return Message.build(false).setMessage("调取第三方话费充值失败");
        }
        XinRongCreateOrderBO xinRongCreateOrderBO = JSON.parseObject(JSON.toJSONString(xinRongChargeBO.getData()), XinRongCreateOrderBO.class);
        response.setRequestId(ThreadContext.getRequestId());
        response.setResult(body);
        response.setPhoneNo(phoneNo);
        response.setMoney(xinRongCreateOrderBO.getMoney());
        response.setOrderNo(orderInfo.getOrderNo().toString());
        response.setOutOrderNo(request.getOrderNo());
        return Message.build().addParam(CommonConstant.RESULT_DTO, response);
    }

    private String getPhoneCarrier(String phone) {
        String carrier = PhoneUtil.getCarrier(phone);
        if (StringUtils.isBlank(carrier)) {
            String substring = StringUtils.substring(phone, 0, 3);
            if (thirdPartyConfigProperties.getMovePhoneParagraph().contains(substring)) {
                return "移动";
            }
            if (thirdPartyConfigProperties.getUmPhoneParagraph().contains(substring)) {
                return "联通";
            }
            if (thirdPartyConfigProperties.getTmPhoneParagraph().contains(substring)) {
                return "电信";
            }
        }
        return carrier;
    }
}
