package com.bxm.thirdparty.platform.before.refund;

import com.bxm.newidea.component.JSON;
import com.bxm.newidea.component.bo.Message;
import com.bxm.newidea.component.tools.StringUtils;
import com.bxm.newidea.component.uuid.SequenceCreater;
import com.bxm.newidea.component.uuid.config.SequenceHolder;
import com.bxm.thirdparty.platform.adapter.context.PlatformContext;
import com.bxm.thirdparty.platform.before.IBeforeThirdPartyPlatformAction;
import com.bxm.thirdparty.platform.context.ThreadContext;
import com.bxm.thirdparty.platform.enums.PaymentOrderStatusEnum;
import com.bxm.thirdparty.platform.enums.PaymentRefundStatusEnum;
import com.bxm.thirdparty.platform.facade.request.PaymentRefundRequest;
import com.bxm.thirdparty.platform.mapper.PaymentOrderV2Mapper;
import com.bxm.thirdparty.platform.mapper.PaymentRefundV2Mapper;
import com.bxm.thirdparty.platform.model.entity.PaymentOrderV2Entity;
import com.bxm.thirdparty.platform.model.entity.PaymentRefundV2Entity;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.util.Date;
import java.util.List;
import java.util.Objects;
import java.util.Optional;

/**
 * @author lowi
 * @date 2023/4/6 11:11
 */
@Component
@Slf4j
public class BeforeRefundAction implements IBeforeThirdPartyPlatformAction<PaymentRefundRequest, PaymentRefundV2Entity> {

    @Resource
    private PaymentOrderV2Mapper paymentOrderV2Mapper;

    @Resource
    private PaymentRefundV2Mapper paymentRefundV2Mapper;

    @Resource
    private SequenceCreater sequenceCreater;

    @Override
    public Class<?> support() {
        return PaymentRefundRequest.class;
    }

    @Override
    public Message validationArgs(PlatformContext<PaymentRefundRequest, PaymentRefundV2Entity> context) {
        PaymentRefundRequest request = context.getRequest();

        String orderNo = request.getOrderNo();
        String outTrackNo = request.getOutTrackNo();
        if (StringUtils.isBlank(orderNo) && StringUtils.isBlank(outTrackNo)) {
            return Message.build(false).setMessage("订单号,和内部封装服务订单号不能同时为空。");
        }

        PaymentOrderV2Entity orderV2Entity = getPaymentOrder(context);

        if (orderV2Entity == null) {
            return Message.build(false).setMessage("未查询到支付成功的订单信息");
        }

        if (Objects.equals(orderV2Entity.getMockOrder(), 1)) {
            return Message.build(false).setMessage("mock订单不支持退款");
        }
        //如果不传退款，则退订单的金额
        if (Objects.isNull(request.getRefundAmount())) {
            request.setRefundAmount(orderV2Entity.getAmount());
        }

        if (orderV2Entity.getAmount().compareTo(request.getRefundAmount()) < 0) {
            return Message.build(false).setMessage("退款金额不能大于支付订单金额");
        }
        PaymentRefundV2Entity refundV2Entity = paymentRefundV2Mapper.getByPaymentOrderNo(orderV2Entity.getOrderNo());

        if (Objects.nonNull(refundV2Entity)) {
            if ((Objects.equals(PaymentRefundStatusEnum.REFUND_ING.getCode(), refundV2Entity.getStatus())
                    || Objects.equals(PaymentRefundStatusEnum.REFUNDED.getCode(), refundV2Entity.getStatus()))) {
                log.warn("订单[{}]已退款或正在退款，不予重复处理", orderNo);
                return Message.build(false, "退款处理中或已处理");
            }
        }
        Integer status = orderV2Entity.getStatus();
        if (PaymentOrderStatusEnum.REFUND_ING.match(status) || PaymentOrderStatusEnum.REFUNDED.match(status)) {
            log.warn("业务订单[{}]处于退款中或已退款状态，仍然被调用", orderNo);
            return Message.build(false, "订单已退款或正在退款中");
        }

        if (!PaymentOrderStatusEnum.SUCCESS.match(status) && !PaymentOrderStatusEnum.REFUND_FAIL.match(status)) {
            log.warn("业务订单[{}]当前处于[{}],不支持退款", orderNo, status);
            return Message.build(false, "订单不处理支付成功的状态，无法退款");
        }

        request.setOrderNo(orderV2Entity.getOrderNo().toString());

        context.setExtStr(JSON.toJSONString(orderV2Entity));
        context.setMock(orderV2Entity.getMockOrder() == 1);
        context.setSync(Boolean.TRUE);
        return Message.build();
    }

    private PaymentOrderV2Entity getPaymentOrder(PlatformContext context) {
        PaymentRefundRequest request = (PaymentRefundRequest) context.getRequest();
        String orderNo = request.getOrderNo();
        PaymentOrderV2Entity orderV2Entity = null;
        if (StringUtils.isNotBlank(orderNo)) {
            orderV2Entity = paymentOrderV2Mapper.getByOrderNo(orderNo);
        } else {
            List<PaymentOrderV2Entity> paymentOrderV2Entities = paymentOrderV2Mapper.getByOutOrderNo(request.getOutTrackNo());
            if (paymentOrderV2Entities.size() > 1) {
                log.warn("业务订单号[{}]对应的订单有多条", request.getOutTrackNo());
            }
            Optional<PaymentOrderV2Entity> paymentOrderV2EntityOptional = paymentOrderV2Entities.stream().filter(paymentOrderV2Entity -> {
                return Objects.equals(PaymentOrderStatusEnum.SUCCESS.getCode(), paymentOrderV2Entity.getStatus());
            }).findFirst();
            orderV2Entity = paymentOrderV2EntityOptional.orElse(null);
        }
        return orderV2Entity;
    }

    @Override
    public void insertInfo(PlatformContext<PaymentRefundRequest, PaymentRefundV2Entity> context) {
        PaymentRefundRequest request = context.getRequest();
        //这步不会为null,上面已经判断过了
        PaymentOrderV2Entity paymentOrder = JSON.parseObject(context.getExtStr(), PaymentOrderV2Entity.class);

        PaymentRefundV2Entity historyInfo = paymentRefundV2Mapper.getByPaymentOrderNo(paymentOrder.getOrderNo());

        PaymentRefundV2Entity refundV2Entity = new PaymentRefundV2Entity();
        refundV2Entity.setId(sequenceCreater.nextLongId());
        refundV2Entity.setPayPlatform(paymentOrder.getPayPlatform());
        refundV2Entity.setAccountId(paymentOrder.getAccountId());
        refundV2Entity.setApplicationName(paymentOrder.getApplicationName());
        refundV2Entity.setAmount(paymentOrder.getAmount());
        refundV2Entity.setStatus(0);
        refundV2Entity.setRemark(paymentOrder.getRemark());
        refundV2Entity.setPaymentOrderNo(paymentOrder.getOrderNo());
        refundV2Entity.setRefundOrderNo(SequenceHolder.nextStringId());
        refundV2Entity.setCreateTime(new Date());
        refundV2Entity.setOutOrderNo(request.getRefundOrderNo());
        refundV2Entity.setRequestId(ThreadContext.getRequestId());
        if (Objects.nonNull(historyInfo)) {
            refundV2Entity.setId(historyInfo.getId());
            paymentRefundV2Mapper.updateById(refundV2Entity);
        } else {
            paymentRefundV2Mapper.insert(refundV2Entity);
        }

        context.setOrderInfo(refundV2Entity);
    }
}
