package com.bxm.kylin._super.sdk;

import com.alibaba.fastjson.JSONObject;
import com.bxm.kylin._super.sdk.modal.*;
import com.bxm.warcar.cache.DataExtractor;
import com.bxm.warcar.cache.Fetcher;
import com.bxm.warcar.utils.KeyBuilder;
import com.bxm.warcar.utils.http.pool.RestTemplatePool;
import com.bxm.warcar.utils.http.properties.HttpProperties;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpMethod;
import org.springframework.web.client.RestTemplate;
import org.springframework.web.util.UriComponents;
import org.springframework.web.util.UriComponentsBuilder;

import java.util.*;

/**
 * @author allen
 * @date 2021-06-08
 * @since 1.0
 */
@Slf4j
public class DefaultKylinApiClientImpl implements KylinApiClient {

    private final Fetcher fetcher;
    private final RestTemplatePool rest;
    private final KylinProperties kylinProperties;

    public DefaultKylinApiClientImpl(Fetcher fetcher, KylinProperties kylinProperties) {
        this.fetcher = fetcher;
        this.rest = new RestTemplatePool(new HttpProperties());
        this.kylinProperties = kylinProperties;
    }

    @Override
    public List<Domain> getAvailableDomains(String groupId, String ua, String ip) {
        return this.getAvailableDomains(groupId, ua, ip, "");
    }

    @Override
    public List<Domain> getAvailableDomains(String groupId, String ua, String ip, String path) {
        Map<String, Object> params = Maps.newHashMap();
        params.put("groupId", groupId);
        params.put("ua", ua);
        params.put("ip", ip);
        params.put("path", path);
        try {
            String url = kylinProperties.getRedirectHost() + "/domains?groupId={groupId}&ua={ua}&ip={ip}&path={path}";
            HttpEntity<String> exchange = this.post(url, String.class, params);
            if (!exchange.hasBody()) {
                return null;
            }
            return JSONObject.parseArray(exchange.getBody(), Domain.class);
        } catch (Exception e) {
            log.error("post: ", e);
            return null;
        }
    }

    @Override
    public List<DomainGroup> getDomainGroups() {
        return fetcher.fetchList(() -> KeyBuilder.build("tmp", "kylin", "domainGroups"), new DataExtractor<List<DomainGroup>>() {
            @Override
            public List<DomainGroup> extract() {
                Map<String, Object> params = Maps.newHashMap();
                params.put("size", 100);
                params.put("current", 1);
                try {
                    String url = kylinProperties.getApiHost() + "/domain-group/page-query?size={size}&current={current}";
                    HttpEntity<DomainGroupPage> entity = get(url, DomainGroupPage.class, params);
                    DomainGroupPage body = entity.getBody();
                    if (Objects.isNull(body)) {
                        return null;
                    }
                    return body.getRecords();
                } catch (Exception e) {
                    log.error("exchange: ", e);
                    return null;
                }
            }
        }, DomainGroup.class, 5 * 60);
    }

    @Override
    public CheckPlan createCheckPlan(Long envId, Long groupId, Long domainId, String sourceUrl) {
        UriComponents build = UriComponentsBuilder.fromUriString(sourceUrl).build();

        String path = build.getPath();

        if (Objects.nonNull(groupId)) {
            CheckPlan exists = getCheckPlanOnCreateJustNow(envId, groupId, domainId, path);
            if (Objects.nonNull(exists)) {
                return exists;
            }
        }

        CheckPlan checkPlan = CheckPlan.builder()
                .environmentId(envId)
                .domainId(domainId)
                .checkPath(path)
                .expectStatusCode(200)
                .cronExpression("0 0/1 * * * ?")
                .status(1)
                .webhook(kylinProperties.getWebhook())
                .build();

        try {
            HashMap<String, String> headers = Maps.newHashMap();
            headers.put("Content-Type", "application/json");
            HttpEntity<CheckPlan> exchange = rest.exchange(kylinProperties.getApiHost() + "/check-plan/put",
                    headers,
                    checkPlan,
                    HttpMethod.PUT,
                    CheckPlan.class);
            CheckPlan res = exchange.getBody();
            if (Objects.nonNull(res)) {
                res.setLastCheckResult(CheckPlan.ENABLE);
            }
            return res;
        } catch (Exception e) {
            log.error("exchange: ", e);
            return null;
        }
    }

    @Override
    public void deleteCheckPlan(Long checkPlanId) {
        try {
            rest.exchange(kylinProperties.getApiHost() + "/check-plan/delete/" + checkPlanId,
                    HttpMethod.DELETE,
                    String.class);
        } catch (Exception e) {
            log.info("{} delete checkPlan occur error: {}", checkPlanId, e.getMessage());
        }
    }

    /**
     * 获取指定条件下刚刚创建，还没有检测过的计划。
     *
     * @param envId EnvId
     * @param groupId Group id
     * @param domainId Domain id
     * @param path Path
     * @return 返回 null 则表示没有或有已检测过的计划
     */
    private CheckPlan getCheckPlanOnCreateJustNow(Long envId, Long groupId, Long domainId, String path) {
        Map<String, Object> params = Maps.newHashMap();
        params.put("size", 100);
        params.put("current", 1);
        params.put("environmentId", envId);
        params.put("groupId", groupId);
        params.put("domainId", domainId);
        params.put("status", "1");
        try {
            String url = kylinProperties.getApiHost() + "/check-plan/page-query?size={size}&current={current}&environmentId={environmentId}&groupId={groupId}&domainId={domainId}&status={status}";
            HttpEntity<CheckPlanPage> entity = get(url, CheckPlanPage.class, params);
            CheckPlanPage body = entity.getBody();
            if (Objects.isNull(body)) {
                return null;
            }
            List<CheckPlan> records = body.getRecords();
            Optional<CheckPlan> exists = records.stream().filter((e) -> e.isNewPlan() && StringUtils.equalsIgnoreCase(e.getCheckPath(), path)).findFirst();
            return exists.orElse(null);
        } catch (Exception e) {
            log.error("exchange: ", e);
            return null;
        }
    }

    private <T> HttpEntity<T> get(String url, Class<T> responseType, Map<String, Object> params) {
        HttpEntity<String> httpEntity = new HttpEntity<>(null, new HttpHeaders());
        RestTemplate restTemplate = rest.builder();
        return restTemplate.exchange(url, HttpMethod.GET, httpEntity, responseType, params);
    }

    private <T> HttpEntity<T> post(String url, Class<T> responseType, Map<String, Object> params) {
        HttpEntity<String> httpEntity = new HttpEntity<>(null, new HttpHeaders());
        RestTemplate restTemplate = rest.builder();
        return restTemplate.exchange(url, HttpMethod.POST, httpEntity, responseType, params);
    }
}
