package com.bxm.kylin._super.sdk;


import com.bxm.kylin._super.sdk.modal.CheckPlan;
import com.bxm.kylin._super.sdk.modal.Domain;
import com.bxm.kylin._super.sdk.modal.Protocol;
import com.bxm.warcar.cache.Fetcher;
import com.bxm.warcar.cache.KeyGenerator;
import com.bxm.warcar.cache.Updater;
import com.bxm.warcar.message.Message;
import com.bxm.warcar.message.MessageSender;
import com.bxm.warcar.message.dingding.DingDingMessageSender;
import com.bxm.warcar.utils.TypeHelper;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.util.CollectionUtils;
import org.springframework.web.util.UriComponentsBuilder;

import java.util.*;
import java.util.function.Consumer;
import java.util.function.Predicate;
import java.util.stream.Collectors;

/**
 * @author allen
 * @date 2021-08-23
 * @since 1.0
 */
@Slf4j
public class DefaultKylinImpl implements Kylin {

    private final KylinApiClient client;
    private final CacheKey cacheKey;
    private final Fetcher fetcher;
    private final Updater updater;
    private final MessageSender messageSender;

    private final boolean noticeIfNecessary;

    public DefaultKylinImpl(KylinApiClient client, CacheKey cacheKey, Fetcher fetcher,
                            Updater updater, KylinProperties kylinProperties) {
        this.client = client;
        this.cacheKey = cacheKey;
        this.fetcher = fetcher;
        this.updater = updater;
        this.messageSender = new DingDingMessageSender(kylinProperties.getDingTalkWebhookUrl());
        this.noticeIfNecessary = kylinProperties.isNoticeIfNecessary();
    }

    @Override
    public void start(String id, String groupId, String url) {
        Map<String, Domain> domains = getDomains(id);
        Domain available = getFirstAvailableDomain(domains);
        if (Objects.isNull(available)) {
            List<Domain> disableList = filterCollect(domains, stringDomainEntry -> !stringDomainEntry.getValue().isAvailable());
            String path = UriComponentsBuilder.fromUriString(url).build().getPath();
            available = getFirstAvailableDomainOnKylin(id, groupId, disableList, path);
        }
        if (Objects.isNull(available)) {
            log.warn("No available domain of ticket '{}'", id);
            this.sendMessage(String.format("ID [%s] 没有微信环境可用的域名", id));
            return;
        }

        // 新的URL
        String newUrl = buildNewUrl(available, url);

        String oldUrl = getUrl(id);
        if (StringUtils.equals(oldUrl, newUrl)) {
            log.info("{} oldUrl: {}, newUrl: {}", id, oldUrl, newUrl);
            return;
        }

        updater.hupdate(cacheKey.hashAvailableUrl(), id, newUrl);
        updater.hupdate(cacheKey.hashDomains(id), TypeHelper.castToString(available.getId()), available);

        this.refreshCheckPlanIfNecessary(id, available, newUrl);

        // 发送替换通知
        log.info("{} sourceUrl: {}, newUrl: {}", id, url, newUrl);
        if (noticeIfNecessary) {
            this.sendMessage(String.format("ID [%s] \n落地页链接\n %s \n当前已替换成\n %s", id, url, newUrl));
        }
    }

    @Override
    public void close(String id) {
        this.close(id, true);
    }

    @Override
    public void close(String id, boolean withDomains) {
        // 删除麒麟监测计划
        CheckPlan checkPlan = fetcher.hfetch(cacheKey.hashCheckPlan(), id, CheckPlan.class);
        if (Objects.nonNull(checkPlan)) {
            client.deleteCheckPlan(checkPlan.getId());
        }

        // 删除配置
        updater.hremove(cacheKey.hashAvailableUrl(), id);
        updater.hremove(cacheKey.hashCheckPlan(), id);
        if (withDomains) {
            updater.remove(cacheKey.hashDomains(id));
        }

        log.info("{} closed.", id);
    }

    @Override
    public void changed(CheckPlan checkPlan) {
        if (Objects.isNull(checkPlan)) {
            return;
        }
        boolean enable = checkPlan.isAvailable();

        Map<String, CheckPlan> map = fetcher.hfetchall(cacheKey.hashCheckPlan(), CheckPlan.class);
        if (MapUtils.isEmpty(map)) {
            return;
        }

        map.entrySet().forEach(new Consumer<Map.Entry<String, CheckPlan>>() {
            @Override
            public void accept(Map.Entry<String, CheckPlan> entry) {
                String id = entry.getKey();
                CheckPlan plan = entry.getValue();
                if (!Objects.equals(checkPlan.getId(), plan.getId())) {
                    return;
                }
                String domainId = TypeHelper.castToString(checkPlan.getDomainId());
                Domain domain = getDomain(id, domainId);
                if (Objects.isNull(domain)) {
                    return;
                }
                domain.setAvailable(enable);
                updater.hupdate(cacheKey.hashDomains(id), domainId, domain);

                log.info("ID {} domain {} has changed state: {}", id, domainId, enable);

                String groupId = domain.getGroupId();
                String url = getUrl(id);
                start(id, groupId, url);
            }
        });
    }

    @Override
    public String getUrl(String id) {
        return fetcher.hfetch(cacheKey.hashAvailableUrl(), id, String.class);
    }

    @Override
    public Map<String, String> getAllUrl() {
        return fetcher.hfetchall(cacheKey.hashAvailableUrl(), String.class);
    }

    @Override
    public Map<String, String> getAllCheckPlan() {
        Map<String, CheckPlan> map = Optional.ofNullable(fetcher.hfetchall(cacheKey.hashCheckPlan(), CheckPlan.class)).orElse(new HashMap<>());
        Map<String, String> rst = Maps.newHashMap();
        map.forEach((key, value) -> rst.put(key, TypeHelper.castToString(value.getId())));
        return rst;
    }

    private List<Domain> filterCollect(Map<String, Domain> domains, Predicate<Map.Entry<String, Domain>> filter) {
        return domains.entrySet()
                .stream()
                .filter(filter)
                .map(Map.Entry::getValue).collect(Collectors.toList());
    }

    private Domain getFirstAvailableDomainOnKylin(String id, String groupId, List<Domain> exclude, String path) {
        List<Domain> domains = client.getAvailableDomains(groupId, Constants.UA_MP, Constants.IP, path);
        if (CollectionUtils.isEmpty(domains)) {
            log.warn("No available domain of group '{}'", groupId);
            this.sendMessage(String.format("域名组 [%s] 没有微信环境可用的域名", groupId));
            return null;
        }
        if (!CollectionUtils.isEmpty(exclude)) {
            Set<Long> excludeIds = exclude.stream().map(Domain::getId).collect(Collectors.toSet());
            domains.removeIf(domain -> excludeIds.contains(domain.getId()));
        }
        if (CollectionUtils.isEmpty(domains)) {
            log.warn("After remove, No available domain of group '{}'", groupId);
            this.sendMessage(String.format("ID [%s] 域名组 [%s] 没有微信环境可用的域名", id, groupId));
            return null;
        }
        int remain = domains.size() - 1;
        if (remain <= 2) {
            log.warn("id [{}] group [{}] available domain quantity less: {}", id, groupId, remain);
            this.sendMessage(String.format("ID [%s] 域名组 [%s] 微信环境剩余可用域名数为 %s", id, groupId, remain));
        }
        return domains.iterator().next();
    }

    /**
     * 获取当前广告券可用的域名id
     * @param domains 域名列表
     * @return domainId
     */
    private Domain getFirstAvailableDomain(Map<String, Domain> domains) {
        // 判断已保存的域名列表是否有可用的，如果有，那么直接结束了。
        if (MapUtils.isNotEmpty(domains)) {
            for (Map.Entry<String, Domain> entry : domains.entrySet()) {
                Domain domain = entry.getValue();
                if (domain.isAvailable()) {
                    return domain;
                }
            }
        }
        return null;
    }

    private Map<String, Domain> getDomains(String id) {
        KeyGenerator hashDomains = cacheKey.hashDomains(id);
        return Optional.ofNullable(fetcher.hfetchall(hashDomains, Domain.class)).orElse(new HashMap<>());
    }

    private Domain getDomain(String id, String domainId) {
        return fetcher.hfetch(cacheKey.hashDomains(id), domainId, Domain.class);
    }

    /**
     * 根据 {@link Domain} 重新构建 {@code sourceUrl}，替换 schema / host 内容，生成一个新的链接并返回。
     * @param domain 域名对象
     * @param sourceUrl 原始链接
     * @return 新的链接。
     */
    private String buildNewUrl(Domain domain, String sourceUrl) {
        return UriComponentsBuilder.fromUriString(sourceUrl)
                .scheme(Protocol.of(domain.getProtocol()).getName())
                .host(domain.getDomain())
                .build()
                .toString();
    }

    /**
     * 如果有必要的话，刷新监测计划。
     *
     * @param id ID
     * @param domain 当前需要监测的域名
     * @param url 监测的URL
     */
    private void refreshCheckPlanIfNecessary(String id, Domain domain, String url) {
        CheckPlan current = fetcher.hfetch(cacheKey.hashCheckPlan(), id, CheckPlan.class);
        if (Objects.nonNull(current)) {
            // 如果当前计划是可用的，并且与当前域名以及监测路径一致则不需要更新。
            String path = UriComponentsBuilder.fromUriString(url).build().getPath();

            boolean alikeDomain = Objects.equals(current.getDomainId(), domain.getId());
            boolean alikePath = StringUtils.equals(current.getCheckPath(), path);

            if (current.isAvailable() && alikeDomain && alikePath) {
                return;
            }
        }

        CheckPlan latest = client.createCheckPlan(Constants.ENV_ID, null, domain.getId(), url);
        if (Objects.isNull(latest)) {
            this.sendMessage(String.format("ID [%s] 创建监测计划失败了！", id));
            return;
        }

        updater.hupdate(cacheKey.hashCheckPlan(), id, latest);

        // 删除旧的计划
        if (Objects.nonNull(current)) {
            client.deleteCheckPlan(current.getId());
        }
    }

    private void sendMessage(String content) {
        try {
            this.messageSender.send2(new Message(content));
        } catch (Exception e) {
            log.warn("send2: ", e);
        }
    }
}
