package com.bxm.component.bus.listener;

import com.bxm.component.bus.event.RemoteEvent;
import com.bxm.component.bus.event.TransferMedia;
import com.bxm.component.bus.subscriber.IRemoteSubscriber;
import com.bxm.newidea.component.JSON;
import com.bxm.newidea.component.tools.ReflectionUtils;
import com.google.common.collect.Maps;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeansException;
import org.springframework.boot.ApplicationArguments;
import org.springframework.boot.ApplicationRunner;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.context.ApplicationListener;
import org.springframework.stereotype.Component;

import java.util.LinkedHashMap;
import java.util.Map;

/**
 * 远程事件监听器
 * 接收到远程事件后，根据事件中的值类型进行对应的分发处理
 *
 * @author liujia
 * @date 7/1/21 6:40 PM
 **/
@Component
@Slf4j
@AllArgsConstructor
public class RemoteEventListener implements ApplicationListener<RemoteEvent>, ApplicationRunner, ApplicationContextAware {

    private ApplicationContext applicationContext;

    private Map<String, IRemoteSubscriber> remoteSubscriberMap;

    private Map<String, Class<?>> subscriberClassMap;

    @Override
    public void onApplicationEvent(RemoteEvent event) {
        if (log.isDebugEnabled()) {
            log.debug("接受到远程事件，事件对象为：{}", JSON.toJSONString(event));
        }

        Object source = event.getTransferObject();
        if (null == source) {
            log.warn("接收到的远程事件，没有传递值，不予处理");
            return;
        }

        if (source instanceof LinkedHashMap) {
            LinkedHashMap dataMap = (LinkedHashMap) source;
            String transferClassName = (String) dataMap.get("className");
            String jsonValue = (String) dataMap.get("jsonValue");

            consume(transferClassName, jsonValue);
        } else if (source instanceof TransferMedia) {
            // 如果跳过当前实例，则不执行
            if (event.isSkipCurrentService()) {
                if (log.isDebugEnabled()) {
                    log.debug("当前实例不执行，跳过：{}", event.getOriginService());
                }
                return;
            }

            TransferMedia media = (TransferMedia) source;

            consume(media.getClassName(), media.getJsonValue());
        } else {
            log.info("接收的对象类型无法处理，类型为：{}", source.getClass().getName());
        }
    }

    private void consume(String className, String jsonValue) {
        IRemoteSubscriber subscriber = remoteSubscriberMap.get(className);

        if (null == subscriber) {
            log.info("[{}]没有对应的订阅者，不进行处理", className);
            return;
        }

        subscriber.consume(JSON.parseObject(jsonValue, subscriberClassMap.get(className)));

    }

    @Override
    public void run(ApplicationArguments args) {
        Map<String, IRemoteSubscriber> subscriberMap = applicationContext.getBeansOfType(IRemoteSubscriber.class);

        remoteSubscriberMap = Maps.newHashMap();
        subscriberClassMap = Maps.newHashMap();

        for (IRemoteSubscriber subscriber : subscriberMap.values()) {
            Class<?> subscriberValueClass = ReflectionUtils.getFirstGenericType(subscriber.getClass());

            if (null == subscriberValueClass) {
                log.error("声明的[{}]没有指定泛型，无法匹配订阅数据", subscriber.getClass().getSimpleName());
                continue;
            }

            remoteSubscriberMap.put(subscriberValueClass.getName(), subscriber);
            subscriberClassMap.put(subscriberValueClass.getName(), subscriberValueClass);
        }

    }

    @Override
    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
        this.applicationContext = applicationContext;
    }
}
