package com.bxm.component.mq.message.send.impl.strategy;

import com.bxm.component.mq.config.RabbitConfigurationProperties;
import com.bxm.component.mq.enums.MessageTypeEnum;
import com.bxm.newidea.component.JSON;
import com.rabbitmq.client.Channel;
import lombok.extern.slf4j.Slf4j;
import org.springframework.amqp.rabbit.connection.Connection;
import org.springframework.amqp.rabbit.connection.ConnectionFactory;
import org.springframework.amqp.rabbit.core.RabbitTemplate;
import org.springframework.amqp.support.converter.Jackson2JsonMessageConverter;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.io.IOException;
import java.util.concurrent.TimeoutException;

/**
 * @author lowi
 * @date 2021/8/3 16:09
 */
@Slf4j
@Component
public class RabbitMessageSendStrategy extends MessageStrategy {

    @Autowired
    private ConnectionFactory connectionFactory;

    @Autowired
    private RabbitTemplate rabbitTemplate;

    @Autowired
    private RabbitConfigurationProperties configurationProperties;

    @Override
    public void sendMessage(String uniqueMark, Object message) {
        String routing = uniqueMark + "-routing";
        String exchange = uniqueMark + "-exchange";
        try {
            buildMessageSender(uniqueMark, routing, exchange, message);
        } catch (IOException e) {
            log.error("发送消息失败，{}", JSON.toJSONString(message), e);
        }
    }

    private void buildMessageSender(String queue, String routing, String exchange, Object message) throws IOException {
        final Connection connection = connectionFactory.createConnection();

        //设置发送的通道信息
        buildQueue(exchange, routing, queue, connection);
        rabbitTemplate.setConnectionFactory(connectionFactory);
        rabbitTemplate.setMandatory(true);
        rabbitTemplate.setExchange(exchange);
        rabbitTemplate.setRoutingKey(routing);
        rabbitTemplate.setMessageConverter(new Jackson2JsonMessageConverter());
        //发送具体的消息
        rabbitTemplate.convertAndSend(message);
    }

    private void buildQueue(String exchange, String routingKey,
                            final String queue, Connection connection) throws IOException {
        Channel channel = connection.createChannel(false);
        channel.exchangeDeclare(exchange, configurationProperties.getType(), configurationProperties.getDurable(),
                configurationProperties.getAutoDelete(), configurationProperties.getArguments());
        channel.queueDeclare(queue, configurationProperties.getDurable(), configurationProperties.getExclusive(),
                configurationProperties.getAutoDelete(), configurationProperties.getArguments());
        channel.queueBind(queue, exchange, routingKey);
        try {
            channel.close();
        } catch (TimeoutException e) {
            log.info("关闭通道超时 ", e);
        }
    }

    @Override
    public String getType() {
        return MessageTypeEnum.RABBIT_MQ.name();
    }
}
