package com.bxm.component.bus.config;

import com.bxm.component.bus.event.IEventValue;
import com.bxm.component.bus.event.RemoteEvent;
import com.bxm.component.bus.event.TransferMedia;
import com.bxm.newidea.component.JSON;
import com.bxm.newidea.component.tools.SpringContextHolder;
import lombok.experimental.UtilityClass;
import org.springframework.cloud.bus.BusProperties;

/**
 * 消息总线调用入口
 *
 * @author liujia
 * @date 7/1/21 5:24 PM
 **/
@UtilityClass
public class ComponentEventBus {

    static BusProperties busProperties;

    /**
     * 发布一个服务内部事件，在任意spring bean中的public方法上声明<code>@EventListener</code>注解即可消费
     * 同时声明<code>@Async</code>可支持异步
     *
     * @param event 事件源，可以是任何对象
     */
    public static <T> void publishInnerEvent(T event) {
        SpringContextHolder.getApplicationContext().publishEvent(event);
    }

    /**
     * 发布一个远程事件到消息队列，所有服务均会接收到，包括当前服务
     * 消费方需要引入事件源项目的facade模块，并且声明一个{@link com.bxm.component.bus.subscriber.IRemoteSubscriber}进行消费
     *
     * @param eventValue 事件中传递的对象，和{@link com.bxm.component.bus.subscriber.IRemoteSubscriber}配合使用
     */
    public static <T extends IEventValue> void publishRemoteEvent(T eventValue) {
        publishRemote(eventValue, null, false);
    }

    /**
     * 发布一个远程事件到消息队列，当前服务实例不进行消费（相同服务的其他实例仍然会消费）
     *
     * @param eventValue 事件中传递的对象
     */
    public static <T extends IEventValue> void publishRemoteEventSkipSelf(T eventValue) {
        publishRemote(eventValue, null, true);
    }

    /**
     * 发布一个远程事件到消息队列，仅可以供指定的服务进行消费
     *
     * @param eventValue         事件传递的值
     * @param destinationService 可消费此服务的队列（服务名称为spring.application.name对应的值）
     */
    public static <T extends IEventValue> void publishRemoteEvent(T eventValue, String destinationService) {
        publishRemote(eventValue, destinationService, true);
    }

    private static <T extends IEventValue> void publishRemote(T eventValue, String destinationService, boolean skipSelf) {
        TransferMedia transferMedia = new TransferMedia();
        transferMedia.setClassName(eventValue.getClass().getName());
        transferMedia.setJsonValue(JSON.toJSONString(eventValue));

        RemoteEvent event = new RemoteEvent(transferMedia, busProperties.getId(), destinationService);
        event.setSkipCurrentService(skipSelf);

        SpringContextHolder.getApplicationContext().publishEvent(event);

    }
}
