package com.bxm.component.bus.config;

import com.bxm.component.bus.annotations.EventBusSubscriber;
import com.bxm.component.bus.event.IEventBusSubscriber;
import com.google.common.collect.Lists;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.config.BeanPostProcessor;
import org.springframework.boot.ApplicationArguments;
import org.springframework.boot.ApplicationRunner;
import org.springframework.core.Ordered;
import org.springframework.core.annotation.AnnotationUtils;
import org.springframework.util.ReflectionUtils;

import java.lang.reflect.Method;
import java.util.List;

/**
 * 解析包含了注解{@link EventBusSubscriber}或者实现了{@link IEventBusSubscriber}的接口的bean
 * 将其注册到eventBus中
 *
 * @author liujia
 * @date 12/2/21 4:49 PM
 **/
public class EventBusSubscriberBeanPostProcessor implements BeanPostProcessor, Ordered, ApplicationRunner {

    private List<Object> subscriberList = Lists.newArrayList();

    @Override
    public int getOrder() {
        return Ordered.LOWEST_PRECEDENCE;
    }

    @Override
    public Object postProcessAfterInitialization(Object bean, String beanName) throws BeansException {
        if (bean instanceof IEventBusSubscriber) {
            subscriberList.add(bean);
        } else {
            Method[] declaredMethods = ReflectionUtils.getAllDeclaredMethods(bean.getClass());

            for (Method declaredMethod : declaredMethods) {
                EventBusSubscriber annotation = AnnotationUtils.findAnnotation(declaredMethod, EventBusSubscriber.class);
                if (annotation != null) {
                    subscriberList.add(bean);
                    break;
                }
            }
        }

        return bean;
    }

    @Override
    public void run(ApplicationArguments args) throws Exception {
        for (Object subscriber : subscriberList) {
            ComponentEventBus.eventBus().register(subscriber);
        }
        subscriberList = null;
    }
}
