package com.bxm.newidea.component.sync.cluster;

import com.alibaba.fastjson.JSON;
import com.bxm.component.httpclient.utils.OkHttpUtils;
import com.bxm.newidea.component.sync.config.SecondLevelCacheConfig;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Value;

import java.util.HashMap;
import java.util.Map;

import static org.apache.commons.lang3.StringUtils.isBlank;

/**
 * 基于spring bus event实现的通知策略
 *
 * @author wzy
 * @date 2020年12月17日23:59:16
 */
@Slf4j
public class SpringEventBusClusterPolicy implements ClusterPolicy {

    private final SecondLevelCacheConfig secondLevelCacheConfig;

    private static final String DEFAULT_SERVER_PORT = "8080";

    private static final String EMPTY_STRING = "";

    private static final String PATH_SEPARATOR = "/";

    /**
     * 应用端口号
     */
    @Value("${server.port:8080}")
    private String port;

    @Value("${server.servlet.context-path:/}")
    private String contextPath;

    public SpringEventBusClusterPolicy(SecondLevelCacheConfig secondLevelCacheConfig) {
        this.secondLevelCacheConfig = secondLevelCacheConfig;
    }


    @Override
    public String name() {
        return "eventBus";
    }

    @Override
    public void publish(Command command) {
        // 使用Spring Cloud Bus消息总线发送更新缓存消息，通过访问env接口去发送事件发送缓存变动消息
        String envUrl = getEnvUrl();

        Map<String, String> params = new HashMap<>();
        params.put("name", secondLevelCacheConfig.getEnvName());
        params.put("value", JSON.toJSONString(command));

        String response = OkHttpUtils.postJsonBody(envUrl, JSON.toJSONString(params), null);
        log.info("info: {}", response);
    }

    /**
     * 获取发送缓存更新事件URL
     *
     * @return 返回时间URL
     */
    private String getEnvUrl() {
        String envUrl = secondLevelCacheConfig.getEnvTemplateUrl();

        String finalPort = this.port;
        String finalContextPath = this.contextPath;

        //填充端口号
        if (isBlank(port)) {
            finalPort = DEFAULT_SERVER_PORT;
        }
        //填充访问前缀
        if (isBlank(finalContextPath)) {
            finalContextPath = EMPTY_STRING;
        } else {
            //如果没有以斜杠开头添加斜杠
            if (!finalContextPath.startsWith(PATH_SEPARATOR)) {
                finalContextPath = PATH_SEPARATOR + finalContextPath;
            }
            //如果以斜杠结尾去掉斜杠
            if (finalContextPath.endsWith(PATH_SEPARATOR)) {
                finalContextPath = finalContextPath.substring(0, finalContextPath.length() - 1);
            }
        }

        envUrl = String.format(envUrl, finalPort, finalContextPath);
        return envUrl;
    }
}
