package com.bxm.newidea.component.sync.config;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;
import com.fasterxml.jackson.databind.exc.InvalidTypeIdException;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.config.BeanDefinition;
import org.springframework.boot.autoconfigure.AutoConfigureBefore;
import org.springframework.cloud.bus.event.RemoteApplicationEvent;
import org.springframework.cloud.bus.event.UnknownRemoteApplicationEvent;
import org.springframework.cloud.bus.jackson.BusJacksonAutoConfiguration;
import org.springframework.cloud.bus.jackson.SubtypeModule;
import org.springframework.cloud.stream.annotation.StreamMessageConverter;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.ClassPathScanningCandidateComponentProvider;
import org.springframework.context.annotation.Configuration;
import org.springframework.core.type.filter.AssignableTypeFilter;
import org.springframework.messaging.Message;
import org.springframework.messaging.converter.AbstractMessageConverter;
import org.springframework.util.ClassUtils;
import org.springframework.util.MimeTypeUtils;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Set;

/**
 * 写这个配置类的原因，是 busJsonConverter这个bean使用了ObjectMapper跟HttpMessageConvert冲突了，导致API接口返回的Date类型数据格式变了。
 * 但原本的BusJacksonMessageConverter不让继承，只能自己copy一下
 * <p>
 * {@link BusJacksonAutoConfiguration}
 *
 * @author gonzo
 * @date 2021-01-25 19:50
 **/
@Configuration
@AutoConfigureBefore({ BusJacksonAutoConfiguration.class })
public class CustomBusJacksonAutoConfiguration {

    @Bean(name = "busJsonConverter")
    @StreamMessageConverter
    public AbstractMessageConverter busJsonConverter() {
        return new BusJacksonMessageConverter(null);
    }
}


/**
 * 没办法 这个类不让继承，只能抄一下了
 */
class BusJacksonMessageConverter extends AbstractMessageConverter
        implements InitializingBean {

    private static final Log log = LogFactory.getLog(BusJacksonMessageConverter.class);

    private static final String DEFAULT_PACKAGE = ClassUtils
            .getPackageName(RemoteApplicationEvent.class);

    private final ObjectMapper mapper;

    private final boolean mapperCreated;

    private String[] packagesToScan = new String[]{DEFAULT_PACKAGE};

    public BusJacksonMessageConverter() {
        this(null);
    }

    @Autowired(required = false)
    public BusJacksonMessageConverter(ObjectMapper objectMapper) {
        super(MimeTypeUtils.APPLICATION_JSON);

        if (objectMapper != null) {
            this.mapper = objectMapper;
            this.mapperCreated = false;
        } else {
            this.mapper = new ObjectMapper();
            this.mapperCreated = true;
        }
    }

    public boolean isMapperCreated() {
        return mapperCreated;
    }

    public void setPackagesToScan(String[] packagesToScan) {
        List<String> packages = new ArrayList<>(Arrays.asList(packagesToScan));
        if (!packages.contains(DEFAULT_PACKAGE)) {
            packages.add(DEFAULT_PACKAGE);
        }
        this.packagesToScan = packages.toArray(new String[0]);
    }

    private Class<?>[] findSubTypes() {
        List<Class<?>> types = new ArrayList<>();
        if (this.packagesToScan != null) {
            for (String pkg : this.packagesToScan) {
                ClassPathScanningCandidateComponentProvider provider = new ClassPathScanningCandidateComponentProvider(
                        false);
                provider.addIncludeFilter(
                        new AssignableTypeFilter(RemoteApplicationEvent.class));

                Set<BeanDefinition> components = provider.findCandidateComponents(pkg);
                for (BeanDefinition component : components) {
                    try {
                        types.add(Class.forName(component.getBeanClassName()));
                    } catch (ClassNotFoundException e) {
                        throw new IllegalStateException(
                                "Failed to scan classpath for remote event classes", e);
                    }
                }
            }
        }
        if (log.isDebugEnabled()) {
            log.debug("Found sub types: " + types);
        }
        return types.toArray(new Class<?>[0]);
    }

    @Override
    protected boolean supports(Class<?> aClass) {
        // This converter applies only to RemoteApplicationEvent and subclasses
        return RemoteApplicationEvent.class.isAssignableFrom(aClass);
    }

    @Override
    public Object convertFromInternal(Message<?> message, Class<?> targetClass,
                                      Object conversionHint) {
        Object result = null;
        try {
            Object payload = message.getPayload();

            if (payload instanceof byte[]) {
                try {
                    result = this.mapper.readValue((byte[]) payload, targetClass);
                } catch (InvalidTypeIdException e) {
                    return new UnknownRemoteApplicationEvent(new Object(), e.getTypeId(), (byte[]) payload);
                }
            } else if (payload instanceof String) {
                try {
                    result = this.mapper.readValue((String) payload, targetClass);
                } catch (InvalidTypeIdException e) {
                    return new UnknownRemoteApplicationEvent(new Object(), e.getTypeId(), ((String) payload).getBytes());
                }
            }
        } catch (Exception e) {
            this.logger.error(e.getMessage(), e);
            return null;
        }
        return result;
    }

    @Override
    public void afterPropertiesSet() throws Exception {
        this.mapper.configure(SerializationFeature.WRITE_DATES_AS_TIMESTAMPS, false);
        this.mapper.registerModule(new SubtypeModule(findSubTypes()));
    }
}
