package com.bxm.newidea.component.sync.provider;

import com.bxm.newidea.component.sync.key.DefaultCacheKeyGenerator;
import com.bxm.newidea.component.sync.vo.MonitorCacheVO;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;

/**
 * @author wzy
 * @version 1.0
 * @date 2020/12/16 8:56 下午
 */
public interface CacheProvider {

    /**
     * 构建通过CacheLoader 构建缓存对象
     * 最大容量Long.MAX_VALUE
     * 默认过期时间半小时
     *
     * @param keyGenerator 缓存key
     * @param cacheLoader  缓存loader
     */
    <T, R> void set(DefaultCacheKeyGenerator keyGenerator, Function<T, R> cacheLoader);


    /**
     * 构建缓存对象
     *
     * @param keyGenerator 缓存key
     * @param cacheLoader  缓存loader
     * @param maximumSize  缓存最大容量
     * @param timeUnit     时间单位
     * @param duration     过期时间
     */
    <T, R> void set(DefaultCacheKeyGenerator keyGenerator, Function<T, R> cacheLoader, long maximumSize, TimeUnit timeUnit, long duration);

    /**
     * 构建缓存对象
     * 默认过期时间为半个小时
     *
     * @param keyGenerator key
     * @param cacheLoader  缓存loader
     * @param maximumSize  缓存最大容量
     */
    <T, R> void set(DefaultCacheKeyGenerator keyGenerator, Function<T, R> cacheLoader, long maximumSize);

    /**
     * 批量获取缓存value, 注意这个方法只返回缓存中已经存在的缓存值
     *
     * @param keyGenerator 缓存key
     * @param subKeys      subKeys
     * @return 缓存value
     */
    <K, V> Map<K, V> get(DefaultCacheKeyGenerator keyGenerator, Collection<K> subKeys);

    /**
     * 根据key获取value
     *
     * @param keyGenerator 缓存key
     * @param subKey       缓存key
     * @return 缓存值
     */
    <K, V> V get(DefaultCacheKeyGenerator keyGenerator, K subKey);

    /**
     * 设置缓存key，已经对应的value
     *
     * @param keyGenerator 缓存key
     * @param subKey       缓存具体key值
     * @param value        缓存具体value值
     */
    <K, V> void set(DefaultCacheKeyGenerator keyGenerator, K subKey, V value);

    /**
     * 缓存key是否存在
     *
     * @param keyGenerator 缓存key
     * @param subKey       缓存key
     * @return 是否存在
     */
    <K, V> boolean exists(DefaultCacheKeyGenerator keyGenerator, K subKey);

    /**
     * 批量插入缓存数据
     *
     * @param keyGenerator 缓存key
     * @param elements     缓存数据map
     */
    <K, V> void set(DefaultCacheKeyGenerator keyGenerator, Map<K, V> elements);

    /**
     * 返回所有subKeys
     *
     * @param keyGenerator 缓存key
     * @return 返回所有缓存key的集合
     */
    <K, V> Collection<K> subKeys(DefaultCacheKeyGenerator keyGenerator);

    /**
     * 移除指定的key
     *
     * @param keyGenerator 缓存key
     * @param subKeys      缓存key
     */
    <K, V> void evict(DefaultCacheKeyGenerator keyGenerator, K... subKeys);

    /**
     * 清除指定key的所有缓存信息
     *
     * @param keyGenerator 缓存key
     */
    <K, V> void clear(DefaultCacheKeyGenerator keyGenerator);

    /**
     * 是否存在某个具体的缓存
     *
     * @param keyGenerator 缓存key
     * @return 是否存在
     */
    boolean existsCache(DefaultCacheKeyGenerator keyGenerator);

    /**
     * 获取缓存监控信息
     *
     * @return 所有缓存监控信息列表
     */
    List<MonitorCacheVO> getMonitorInfo();

    /**
     * 清除分组缓存信息
     *
     * @param groupName 分组名称
     */
    void clearGroup(String groupName);

    /**
     * 获取所有的分组名称
     */
    List<String> getAllGroup();
}