package com.bxm.newidea.component.sync.provider.guava;

import com.bxm.newidea.component.sync.key.DefaultCacheKeyGenerator;
import com.bxm.newidea.component.sync.provider.CacheProvider;
import com.bxm.newidea.component.sync.vo.MonitorCacheVO;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.CacheStats;
import com.google.common.cache.LoadingCache;

import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;

/**
 * GuavaLoading Cache Provider，持有所有Guava缓存对象
 *
 * @author wzy
 * @version 1.0
 * @date 2020/12/17 9:15 上午
 */
public class GuavaLoaderProvider implements CacheProvider {

    private Map<String, LoadingCache> caches = new ConcurrentHashMap<>();

    /**
     * 创建一个新的LoadingCache对象
     *
     * @return 返回封装的Guava Cache对象
     */
    public <T, R> LoadingCache<T, R> newGuavaCache(CacheLoader<T, R> cacheLoader,
                                                   long maximumSize,
                                                   TimeUnit timeUnit,
                                                   long duration) {
        return CacheBuilder.newBuilder()
                .recordStats()
                .maximumSize(maximumSize)
                .expireAfterWrite(duration, timeUnit)
                .build(cacheLoader);
    }

    /**
     * 创建一个新的LoadingCache对象
     *
     * @return 返回封装的Guava Cache对象
     */
    public <T, R> LoadingCache<T, R> newGuavaCache(CacheLoader<T, R> cacheLoader) {
        return CacheBuilder.newBuilder()
                .recordStats()
                .build(cacheLoader);
    }

    @Override
    public <T, R> void set(DefaultCacheKeyGenerator keyGenerator, Function<T, R> cacheLoader) {
        caches.computeIfAbsent(keyGenerator.gen(), v -> newGuavaCache(new CacheLoader<T, R>() {
            @Override
            public R load(T key) throws Exception {
                return cacheLoader.apply(key);
            }
        }));
    }

    @Override
    public <T, R> void set(DefaultCacheKeyGenerator keyGenerator,
                           Function<T, R> cacheLoader,
                           long maximumSize,
                           TimeUnit timeUnit,
                           long duration) {
        caches.computeIfAbsent(keyGenerator.gen(), v -> newGuavaCache(new CacheLoader<T, R>() {
            @Override
            public R load(T key) throws Exception {
                return cacheLoader.apply(key);
            }
        }, maximumSize, timeUnit, duration));
    }

    @Override
    public <T, R> void set(DefaultCacheKeyGenerator keyGenerator, Function<T, R> cacheLoader, long maximumSize) {
        caches.computeIfAbsent(keyGenerator.gen(), v -> newGuavaCache(new CacheLoader<T, R>() {
            @Override
            public R load(T key) throws Exception {
                return cacheLoader.apply(key);
            }
        }, maximumSize, TimeUnit.SECONDS, 60 * 30));
    }

    @Override
    public <K, V> Map<K, V> get(DefaultCacheKeyGenerator keyGenerator, Collection<K> subKeys) {
        LoadingCache<K, V> loadingCache = caches.get(keyGenerator.gen());
        if (Objects.isNull(loadingCache)) {
            return null;
        }
        return loadingCache.getAllPresent(subKeys);
    }

    @Override
    public <K, V> V get(DefaultCacheKeyGenerator keyGenerator, K subKey) {
        LoadingCache<K, V> loadingCache = caches.get(keyGenerator.gen());
        if (Objects.isNull(loadingCache)) {
            return null;
        }
        return loadingCache.getUnchecked(subKey);
    }

    @Override
    public <K, V> void set(DefaultCacheKeyGenerator keyGenerator, K subKey, V value) {
        LoadingCache<K, V> loadingCache = caches.get(keyGenerator.gen());
        if (Objects.isNull(loadingCache)) {
            return;
        }
        loadingCache.put(subKey, value);
    }

    @Override
    public <K, V> boolean exists(DefaultCacheKeyGenerator keyGenerator, K subKey) {
        LoadingCache<K, V> loadingCache = caches.get(keyGenerator.gen());
        if (Objects.isNull(loadingCache)) {
            return false;
        }
        return Objects.nonNull(loadingCache.getIfPresent(subKey));
    }

    @Override
    public <K, V> void set(DefaultCacheKeyGenerator keyGenerator, Map<K, V> elements) {
        LoadingCache<K, V> loadingCache = caches.get(keyGenerator.gen());
        if (Objects.isNull(loadingCache)) {
            return;
        }
        loadingCache.putAll(elements);
    }

    @Override
    public <K, V> Collection<K> subKeys(DefaultCacheKeyGenerator keyGenerator) {
        LoadingCache<K, V> loadingCache = caches.get(keyGenerator.gen());
        if (Objects.isNull(loadingCache)) {
            return Collections.emptyList();
        }
        return loadingCache.asMap().keySet();
    }

    @Override
    public <K, V> void evict(DefaultCacheKeyGenerator keyGenerator, K... subKeys) {
        LoadingCache<K, V> loadingCache = caches.get(keyGenerator.gen());
        if (Objects.isNull(loadingCache)) {
            return;
        }
        loadingCache.invalidateAll(Arrays.asList(subKeys));
    }

    @Override
    public <K, V> void clear(DefaultCacheKeyGenerator keyGenerator) {
        LoadingCache<K, V> loadingCache = caches.get(keyGenerator.gen());
        if (Objects.isNull(loadingCache)) {
            return;
        }
        loadingCache.invalidateAll();
    }

    @Override
    public boolean existsCache(DefaultCacheKeyGenerator keyGenerator) {
        return Objects.nonNull(keyGenerator);
    }

    @Override
    public List<MonitorCacheVO> getMonitorInfo() {
        List<MonitorCacheVO> monitorCacheVOList = new ArrayList<>();
        caches.forEach((k, v) -> {
            CacheStats cacheStats = v.stats();
            MonitorCacheVO cacheVO = new MonitorCacheVO();
            cacheVO.setCacheName(k);
            cacheVO.setHintCount(cacheStats.hitCount());
            cacheVO.setMissCount(cacheStats.missCount());
            cacheVO.setHintRate(cacheStats.hitRate());
            cacheVO.setEvictionCount(cacheStats.evictionCount());
            cacheVO.setLoadSuccessCount(cacheStats.loadExceptionCount());
            cacheVO.setLoadExceptionCount(cacheStats.loadSuccessCount());
            cacheVO.setTotalLoadTime(cacheStats.totalLoadTime());
            cacheVO.setRequestCount(cacheStats.requestCount());
            cacheVO.setSize(v.size());
            monitorCacheVOList.add(cacheVO);
        });
        return monitorCacheVOList;
    }

    @Override
    public void clearGroup(String groupName) {
        caches.forEach((k, v) -> {
            String[] keys = k.split(DefaultCacheKeyGenerator.JOIN_CHAR);
            String targetGroupName = keys[0];

            if (Objects.equals(targetGroupName, groupName)) {
                v.invalidateAll();
            }
        });
    }

    @Override
    public List<String> getAllGroup() {
        Set<String> groupNameSet = new HashSet<>();

        caches.forEach((k, v) -> {
            String[] keys = k.split(DefaultCacheKeyGenerator.JOIN_CHAR);
            String targetGroupName = keys[0];

            groupNameSet.add(targetGroupName);
        });

        List<String> resultList = new ArrayList<>(groupNameSet);

        resultList.sort(String::compareTo);
        return resultList;
    }
}