package com.bxm.newidea.component.sync.listener;

import com.bxm.newidea.component.JSON;
import com.bxm.newidea.component.sync.cluster.Command;
import com.bxm.newidea.component.sync.config.SecondLevelCacheConfig;
import com.bxm.newidea.component.sync.constants.CommandTypeEnum;
import com.bxm.newidea.component.sync.core.CacheHolder;
import com.bxm.newidea.component.sync.key.SyncCacheKey;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.cloud.bus.event.EnvironmentChangeRemoteApplicationEvent;
import org.springframework.context.event.EventListener;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.util.Map;

/**
 * 监听spring cloud 刷新消息总线
 *
 * @author wzy
 * @version 1.0
 * @date 2020/10/20 3:40 下午
 */
@Slf4j
@Component
public class SecondLevelCacheListener {

    @Resource
    private SecondLevelCacheConfig secondLevelCacheConfig;

    /**
     * 缓存对象
     */
    private final CacheHolder cacheHolder;

    public SecondLevelCacheListener(CacheHolder cacheHolder) {
        this.cacheHolder = cacheHolder;
    }

    /**
     * 监听当有新的消息时触发的事件
     *
     * @param event 事件对象
     */
    @EventListener
    public void onEnvironmentChangeRemoteApplicationEvent(EnvironmentChangeRemoteApplicationEvent event) {
        // 监听到修改缓存值的消息
        Map<String, String> eventMap = event.getValues();

        log.info("Received Bus event content: {}", eventMap);

        if (StringUtils.isNotBlank(eventMap.get(secondLevelCacheConfig.getEnvName()))) {
            String command = eventMap.get(secondLevelCacheConfig.getEnvName());

            log.info("Received Bus event command: {}", command);

            //将命令反序列化
            if (StringUtils.isNotBlank(command)) {

                log.info("SpringCloudBus Second Level Cache notice command info: {} ", command);

                Command refreshCommand = JSON.parseObject(command, Command.class);
                if (null != refreshCommand) {
                    executeCommand(refreshCommand);
                }
            }
        }
        //打印spring cloud bus事件日志
        log.info(" Source : {} , originService : {} , destinationService : {} \n",
                event.getSource(),
                event.getOriginService(),
                event.getDestinationService());
    }

    /**
     * 执行缓存通知命令
     *
     * @param targetCommand 要执行的目标命令
     */
    private void executeCommand(Command targetCommand) {
        SyncCacheKey refreshKeyGenerator = targetCommand.getKeyGenerator();

        //首先判断是否存在这个缓存
        if (cacheHolder.existsCache(refreshKeyGenerator)) {
            if (targetCommand.getOperator() == CommandTypeEnum.OPT_EVICT_KEY.getType()) {
                for (Integer commandKey : targetCommand.getKeys()) {
                    if (cacheHolder.exists(refreshKeyGenerator, commandKey)) {
                        cacheHolder.evict(refreshKeyGenerator, commandKey);
                    }
                }
            } else if (targetCommand.getOperator() == CommandTypeEnum.OPT_CLEAR_KEY.getType()) {
                cacheHolder.clear(refreshKeyGenerator);
            }
        }
    }
}