package com.bxm.newidea.component.sync.provider.guava;

import com.bxm.newidea.component.sync.key.SyncCacheKey;
import com.bxm.newidea.component.sync.provider.CacheProvider;
import com.bxm.newidea.component.sync.vo.MonitorCacheVO;
import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheStats;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;

import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;

/**
 * GuavaLoading Cache Provider，持有所有Guava缓存对象
 *
 * @author wzy
 * @version 1.0
 * @date 2020/12/17 9:15 上午
 */
public class GuavaLoaderProvider implements CacheProvider {

    private Map<String, Cache> caches = new ConcurrentHashMap<>();

    private Map<String, Function> loaderMap = new ConcurrentHashMap<>();

    /**
     * 创建一个新的LoadingCache对象
     *
     * @return 返回封装的Guava Cache对象
     */
    private <R> Cache<Integer, R> newGuavaCache(long maximumSize,
                                                TimeUnit timeUnit,
                                                long duration) {
        return CacheBuilder.newBuilder()
                .recordStats()
                .maximumSize(maximumSize)
                .expireAfterWrite(duration, timeUnit)
                .build();

    }

    /**
     * 创建一个新的LoadingCache对象
     *
     * @return 返回封装的Guava Cache对象
     */
    private <R> Cache<Integer, R> newGuavaCache() {
        return CacheBuilder.newBuilder()
                .recordStats()
                .build();
    }

    @Override
    public <T, R> void set(SyncCacheKey keyGenerator, Function<T, R> cacheLoader) {
        caches.putIfAbsent(keyGenerator.gen(), newGuavaCache());
        loaderMap.putIfAbsent(keyGenerator.gen(), cacheLoader);
    }

    @Override
    public <T, R> void set(SyncCacheKey keyGenerator,
                           Function<T, R> cacheLoader,
                           long maximumSize,
                           TimeUnit timeUnit,
                           long duration) {
        caches.putIfAbsent(keyGenerator.gen(), newGuavaCache(maximumSize, timeUnit, duration));
        loaderMap.putIfAbsent(keyGenerator.gen(), cacheLoader);
    }

    @Override
    public <T, R> void set(SyncCacheKey keyGenerator, Function<T, R> cacheLoader, long maximumSize) {
        caches.putIfAbsent(keyGenerator.gen(), newGuavaCache(maximumSize, TimeUnit.SECONDS, 60 * 30));
        loaderMap.putIfAbsent(keyGenerator.gen(), cacheLoader);
    }

    @Override
    public <K, V> Map<K, V> get(SyncCacheKey keyGenerator, Collection<K> subKeys) {
        Cache<Integer, V> loadingCache = caches.get(keyGenerator.gen());
        if (Objects.isNull(loadingCache)) {
            return null;
        }

        Map<K, V> result = Maps.newHashMap();

        for (K subKey : subKeys) {
            result.put(subKey, get(keyGenerator, subKey));
        }

        return result;
    }

    @Override
    public <K, V> V get(SyncCacheKey keyGenerator, K subKey) {
        Cache<Integer, V> loadingCache = caches.get(keyGenerator.gen());
        if (Objects.isNull(loadingCache)) {
            return null;
        }

        Integer hashCodeKey = subKey.hashCode();
        V value = loadingCache.getIfPresent(hashCodeKey);
        if (null == value) {
            value = (V) loaderMap.get(keyGenerator.gen()).apply(subKey);
            loadingCache.put(hashCodeKey, value);
        }

        return value;
    }

    @Override
    public <K, V> void set(SyncCacheKey keyGenerator, K subKey, V value) {
        Cache<Integer, V> loadingCache = caches.get(keyGenerator.gen());
        if (Objects.isNull(loadingCache) || subKey == null || value == null) {
            return;
        }
        loadingCache.put(subKey.hashCode(), value);
    }

    @Override
    public <K, V> boolean exists(SyncCacheKey keyGenerator, K subKey) {
        Cache<K, V> loadingCache = caches.get(keyGenerator.gen());
        if (Objects.isNull(loadingCache) || null == subKey) {
            return false;
        }
        Integer subKeyHash = subKey.hashCode();
        return Objects.nonNull(loadingCache.getIfPresent(subKeyHash));
    }

    @Override
    public <K, V> void set(SyncCacheKey keyGenerator, Map<K, V> elements) {
        Cache<Integer, V> loadingCache = caches.get(keyGenerator.gen());
        if (Objects.isNull(loadingCache) || null == elements) {
            return;
        }

        elements.forEach((k, v) -> loadingCache.put(k.hashCode(), v));
    }

    @Override
    public <K, V> void evict(SyncCacheKey keyGenerator, K... subKeys) {
        Cache<K, V> loadingCache = caches.get(keyGenerator.gen());
        if (Objects.isNull(loadingCache) || null == subKeys || subKeys.length == 0) {
            return;
        }

        List<Integer> subKeyHashList = Lists.newArrayList();
        for (K subKey : subKeys) {
            subKeyHashList.add(subKey.hashCode());
        }

        loadingCache.invalidateAll(subKeyHashList);
    }

    @Override
    public <K, V> void clear(SyncCacheKey keyGenerator) {
        Cache<Integer, V> loadingCache = caches.get(keyGenerator.gen());
        if (Objects.isNull(loadingCache)) {
            return;
        }
        loadingCache.invalidateAll();
    }

    @Override
    public boolean existsCache(SyncCacheKey keyGenerator) {
        return caches.get(keyGenerator.gen()) != null;
    }

    @Override
    public List<MonitorCacheVO> getMonitorInfo() {
        List<MonitorCacheVO> monitorCacheVOList = new ArrayList<>();
        caches.forEach((k, v) -> {
            CacheStats cacheStats = v.stats();
            MonitorCacheVO cacheVO = new MonitorCacheVO();
            cacheVO.setCacheName(k);
            cacheVO.setHintCount(cacheStats.hitCount());
            cacheVO.setMissCount(cacheStats.missCount());
            cacheVO.setHintRate(cacheStats.hitRate());
            cacheVO.setEvictionCount(cacheStats.evictionCount());
            cacheVO.setLoadSuccessCount(cacheStats.loadExceptionCount());
            cacheVO.setLoadExceptionCount(cacheStats.loadSuccessCount());
            cacheVO.setTotalLoadTime(cacheStats.totalLoadTime());
            cacheVO.setRequestCount(cacheStats.requestCount());
            cacheVO.setSize(v.size());
            monitorCacheVOList.add(cacheVO);
        });
        return monitorCacheVOList;
    }

    @Override
    public void clearGroup(String groupName) {
        caches.forEach((k, v) -> {
            String[] keys = k.split(SyncCacheKey.JOIN_CHAR);
            String targetGroupName = keys[0];

            if (Objects.equals(targetGroupName, groupName)) {
                v.invalidateAll();
            }
        });
    }

    @Override
    public List<String> getAllGroup() {
        Set<String> groupNameSet = new HashSet<>();

        caches.forEach((k, v) -> {
            String[] keys = k.split(SyncCacheKey.JOIN_CHAR);
            String targetGroupName = keys[0];

            groupNameSet.add(targetGroupName);
        });

        List<String> resultList = new ArrayList<>(groupNameSet);

        resultList.sort(String::compareTo);
        return resultList;
    }
}