package com.bxm.component.elasticjob.config;

import com.bxm.component.elasticjob.job.IShardingElasticJob;
import com.bxm.component.elasticjob.job.ISimplifyElasticJob;
import com.bxm.newidea.component.tools.StringUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.shardingsphere.elasticjob.api.JobConfiguration;
import org.apache.shardingsphere.elasticjob.lite.api.bootstrap.impl.ScheduleJobBootstrap;
import org.apache.shardingsphere.elasticjob.reg.base.CoordinatorRegistryCenter;
import org.springframework.beans.BeansException;
import org.springframework.boot.ApplicationArguments;
import org.springframework.boot.ApplicationRunner;
import org.springframework.boot.autoconfigure.AutoConfigureAfter;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.autoconfigure.data.elasticsearch.ElasticsearchDataAutoConfiguration;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;

import java.util.Map;

/**
 * 初始化配置自定义的任务信息
 *
 * @author liujia
 * @date 6/8/21 6:59 PM
 **/
@Slf4j
@AutoConfigureAfter(ElasticsearchDataAutoConfiguration.class)
@ConditionalOnProperty(prefix = "elasticjob.reg-center", name = "server-lists")
@EnableConfigurationProperties(ComponentElasticJobProperties.class)
public class ComponentElasticJobConfiguration implements ApplicationRunner, ApplicationContextAware {

    private ApplicationContext applicationContext;

    private CoordinatorRegistryCenter coordinatorRegistryCenter;

    private ComponentElasticJobProperties componentElasticJobProperties;

    public ComponentElasticJobConfiguration(ApplicationContext applicationContext,
                                            CoordinatorRegistryCenter coordinatorRegistryCenter,
                                            ComponentElasticJobProperties componentElasticJobProperties) {
        this.applicationContext = applicationContext;
        this.coordinatorRegistryCenter = coordinatorRegistryCenter;
        this.componentElasticJobProperties = componentElasticJobProperties;
    }

    @Override
    public void run(ApplicationArguments args) throws Exception {

        if (componentElasticJobProperties.isEnableMock()) {
            log.info("配置了[component.elasticjob.enableMock],不启动定时任务");
            return;
        }

        // 获取所有的定时任务，启动调度
        Map<String, ISimplifyElasticJob> definedJobMap = applicationContext.getBeansOfType(ISimplifyElasticJob.class);

        log.info("扫描到[{}]个定时任务", definedJobMap.size());

        for (ISimplifyElasticJob job : definedJobMap.values()) {
            if (null == job.cron()) {
                throw new IllegalArgumentException(job.getClass() + "未配置CRON表达式");
            }

            if (StringUtils.isBlank(job.description())) {
                throw new IllegalArgumentException(job.getClass() + " 未配置定时任务description描述");
            }

            JobConfiguration.Builder builder;
            if (job instanceof IShardingElasticJob) {
                IShardingElasticJob shardingElasticJob = (IShardingElasticJob) job;
                builder = JobConfiguration.newBuilder(job.jobName(), shardingElasticJob.shardingTotalCount());
                builder.shardingItemParameters(shardingElasticJob.shardingItemParameters());
            } else {
                builder = JobConfiguration.newBuilder(job.jobName(), 1);
            }

            JobConfiguration configuration = builder
                    .cron(job.cron())
                    .overwrite(true)
                    .failover(true)
                    .description(job.description())
                    .build();

            ScheduleJobBootstrap bootstrap = new ScheduleJobBootstrap(coordinatorRegistryCenter, job, configuration);

            log.debug("创建定时任务：{}", job.jobName());
            bootstrap.schedule();
        }
    }

    @Override
    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
        this.applicationContext = applicationContext;
    }
}
