package com.bxm.newidea.component.geo.service.impl;

import cn.hutool.http.HttpUtil;
import com.bxm.newidea.component.JSONObject;
import com.bxm.newidea.component.config.GeoProperties;
import com.bxm.newidea.component.geo.dto.AddressComponentDTO;
import com.bxm.newidea.component.geo.dto.CoordinateDTO;
import com.bxm.newidea.component.geo.param.GeoPoint;
import com.bxm.newidea.component.geo.service.GeoService;
import com.bxm.newidea.component.tools.GouldUtils;
import com.google.common.collect.Maps;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import java.util.HashMap;
import java.util.Map;
import java.util.Objects;

/**
 * 地理位置相关接口实现
 */
@Service
@Slf4j
@AllArgsConstructor
public class GeoServiceImpl implements GeoService {

    private final GeoProperties geoProperties;

    @Override
    public Double getDistance(GeoPoint origins, GeoPoint destination) {
        if (isCoordinateEmpty(origins, destination)) {
            return -1D;
        }
        //1、通过经纬度获取距离(单位：米)
        return GouldUtils.getDistance(origins.getLat(), origins.getLng(), destination.getLat(), destination.getLng());
    }

    /**
     * 根据地址获取经纬度信息 填写结构化地址信息:省份＋城市＋区县＋城镇＋乡村＋街道＋门牌号码
     *
     * @param address 详细地址
     * @return LocationInfoDTO 地址包装类
     */
    @Override
    public CoordinateDTO getCoordinate(String address) {
        if (Objects.isNull(address)) {
            return null;
        }

        Map<String, Object> map = Maps.newHashMap();
        map.put("key", geoProperties.getKey());
        map.put("address", address);
        String json = HttpUtil.get(geoProperties.getGeoUrl(), map);
        if (Objects.isNull(json)) {
            return null;
        }

        JSONObject jsonObject = JSONObject.parseObject(json);
        if (Objects.isNull(jsonObject)) {
            return null;
        }

        CoordinateDTO coordinate = null;
        String status = jsonObject.getString("status");
        String count = jsonObject.getString("count");
        if ("1".equalsIgnoreCase(status) && "1".equals(count)) {
            JSONObject geocode = jsonObject.getJSONArray("geocodes").getJSONObject(0);
            String location = geocode.getString("location");
            String[] coordinateArray = location.split(",");
            coordinate = new CoordinateDTO();
            coordinate.setLongitude(Double.valueOf(coordinateArray[0]));
            coordinate.setLatitude(Double.valueOf(coordinateArray[1]));
        }
        return coordinate;
    }

    /***
     * 判断是否在（多少米）范围内
     * @param origins 起始点
     * @param destination 目的地
     * @param radius 匹配的半径（单位米）
     * @return 返回true、false
     */
    @Override
    public Boolean isWithinTheScopeOf(GeoPoint origins, GeoPoint destination, Long radius) {
        if (isCoordinateEmpty(origins, destination)) {
            return false;
        }
        return GouldUtils.isWithinTheScopeOf(
            origins.getLng(),
            origins.getLat(),
            destination.getLng(),
            destination.getLat(),
            radius
        );
    }

    /**
     * 判断是否在某个圆形区域内
     *
     * @param origins 起始经纬度
     * @param area1   区域经纬度1
     * @param area2   区域经纬度2
     * @return 是否在范围内 false、true
     */
    @Override
    public Boolean isInArea(GeoPoint origins, GeoPoint area1, GeoPoint area2) {
        return GouldUtils.isInArea(
            origins.getLat(),
            origins.getLng(),
            area1.getLat(),
            area2.getLat(),
            area1.getLng(),
            area2.getLng()
        );
    }

    /***
     * 根据经纬度信息 获取实际的地理位置
     * @param point 经纬度信息
     * @return 返回经纬度获取地址对象 {@link AddressComponentDTO}
     */
    @Override
    public AddressComponentDTO getAddress(GeoPoint point) {

        HashMap<String, Object> map = Maps.newHashMap();
        map.put("output", "JSON");
        map.put("location", point.getLng() + "," + point.getLat());
        map.put("key", geoProperties.getKey());
        map.put("radius", 0);
        map.put("extensions", "base");

        //请求地理位置
        String json = HttpUtil.get(geoProperties.getRegeoUrl(), map);
        if (Objects.isNull(json)) {
            return null;
        }

        //返回集格式转换json、判断返回状态是否成功
        JSONObject jsonObject = JSONObject.parseObject(json);
        if (null == jsonObject || 1 != jsonObject.getInteger("status")) {
            log.warn("解析经纬度信息失败，请求参数：{},响应结果：{}", point, jsonObject);
            return null;
        }

        //解析返回地理位置参数
        JSONObject regexCode = jsonObject.getJSONObject("regeocode");
        if (regexCode.size() <= 0) {
            return null;
        }

        // 结构化地址信息 结构化地址信息包括：省份＋城市＋区县＋城镇＋乡村＋街道＋门牌号码
        //如果坐标点处于海域范围内，则结构化地址信息为：省份＋城市＋区县＋海域信息
        String formatted = regexCode.getString("formatted_address");

        //行政区划具体信息 地址元素列表
        JSONObject addressComponent = regexCode.getJSONObject("addressComponent");
        AddressComponentDTO dto = JSONObject.parseObject(addressComponent.toJSONString(), AddressComponentDTO.class);
        if (Objects.isNull(dto)){
            dto=new AddressComponentDTO();
        }
        dto.setLat(point.getLat());
        dto.setLng(point.getLng());
        dto.setFormattedAddress(formatted);
        return dto;
    }

    private boolean isCoordinateEmpty(GeoPoint origins, GeoPoint destination) {
        if (Objects.isNull(origins) || Objects.isNull(destination)) {
            return true;
        }
        return origins.getLat() == null
            || origins.getLng() == null
            || destination.getLat() == null
            || destination.getLng() == null;
    }
}
