package com.bxm.component.graceful.shutdown;

import com.bxm.component.graceful.shutdown.config.ShutdownProperties;
import com.bxm.component.graceful.shutdown.event.ShutdownEvent;
import lombok.extern.slf4j.Slf4j;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ConfigurableApplicationContext;
import org.springframework.context.support.AbstractApplicationContext;
import org.springframework.util.Assert;
import org.springframework.util.ReflectionUtils;

import java.lang.reflect.Field;

/**
 * 魔改 spring application shutdown hook
 * 将ApplicationContext的shutdown hook修改为自定义的hook，并在接受到kill -15 的signal时，发送{@link ShutdownEvent}
 * 可以在spring shutdown执行前处理自定义的shutdown逻辑
 *
 * 只需要响应{@link ShutdownEvent}事件即可。
 * @author Gonzo
 * @date 2019-12-04 15:46
 */
@Slf4j
public class ShutdownHook extends Thread {

    private ApplicationContext applicationContext;

    private ShutdownProperties shutdownProperties;

    public ShutdownHook(ApplicationContext applicationContext, ShutdownProperties shutdownProperties) {
        this.applicationContext = applicationContext;
        this.shutdownProperties = shutdownProperties;
        // 注册shutdown hook
        registerShutdownHook();
    }

    /**
     * 将默认的application context的shutdown hook设置为自定义的
     */
    private void registerShutdownHook() {
        log.info("注册自定义 shutdown hook");
        try {
            Field shutdownHook = ReflectionUtils.findField(AbstractApplicationContext.class, "shutdownHook");
            ReflectionUtils.makeAccessible(shutdownHook);
            shutdownHook.set(this.applicationContext, this);

        } catch (IllegalAccessException e) {
            log.error("自定义 shutdownHook 失败", e);
        }
        Runtime.getRuntime().addShutdownHook(this);
    }

    /**
     * 自定义的shutdown实现
     * 在原本的shutdown前，调用自定义的shutdown
     */
    @Override
    public void run() {
        log.info("运行自定义 shutdown hook");
        applicationContext.publishEvent(new ShutdownEvent(applicationContext));

        try {
            // 阻塞
            Thread.sleep(shutdownProperties.getAfterCustomEventFinishWaitTimeSecond() * 1000);
        } catch (InterruptedException e) {
            log.error("等待 自定义shutdown hook阻塞被中断");
        }

        log.info("运行spring shutdown hook");
        Assert.isInstanceOf(ConfigurableApplicationContext.class, applicationContext);
        ((ConfigurableApplicationContext) applicationContext).close();
    }
}
