package com.bxm.component.httpclient.service;

import com.bxm.newidea.component.service.BaseService;
import com.bxm.newidea.component.tools.FileUtils;
import okhttp3.*;
import org.apache.http.client.utils.URIBuilder;
import org.apache.http.entity.ContentType;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.URISyntaxException;
import java.util.Map;
import java.util.Objects;

@Service
public class OKHttpService extends BaseService {

    private static final Logger log = LoggerFactory.getLogger(OKHttpService.class);

    /**
     * 发起get请求
     * @param url 请求地址
     * @return 响应结果
     */
    public String get(String url, Map<String, String> headerMap) {
        return extractResponse(new Request.Builder().url(url).get(), headerMap);
    }

    private String extractResponse(Request.Builder requestBuilder, Map<String, String> headerMap) {
        try {
            for (Map.Entry<String, String> entry : headerMap.entrySet()) {
                requestBuilder.addHeader(entry.getKey(), entry.getValue());
            }
            Request request = requestBuilder.build();

            OkHttpClient client = new OkHttpClient();
            Response response = client.newCall(request).execute();
            if (response.isSuccessful()) {
                return Objects.requireNonNull(response.body()).string();
            }
        } catch (IOException e) {
            log.error(e.getMessage(), e);
        }
        return null;
    }

    public String get(String url, Map<String, String> params, Map<String, String> headerMap) {
        Request.Builder request = null;
        try {
            URIBuilder uriBuilder = new URIBuilder(url);
            for (Map.Entry<String, String> entry : params.entrySet()) {
                uriBuilder.addParameter(entry.getKey(), entry.getValue());
            }

            request = new Request.Builder()
                    .url(uriBuilder.toString())
                    .get();

        } catch (URISyntaxException e) {
            log.error(e.getMessage(), e);
        }
        return extractResponse(request, headerMap);
    }

    /**
     * post提交form表单
     * @param url    提交地址
     * @param params 表单参数KV
     * @return 响应结果
     */
    public String postForm(String url, Map<String, String> params, Map<String, String> headerMap) {
        FormBody.Builder builder = new FormBody.Builder();

        for (Map.Entry<String, String> entry : params.entrySet()) {
            builder.add(entry.getKey(), entry.getValue());
        }

        Request.Builder request = new Request.Builder()
                .url(url)
                .post(builder.build());

        return extractResponse(request, headerMap);
    }

    /**
     * 提交空白的body post请求
     * @param url     请求地址
     * @param content 请求body内容
     * @return 响应结果
     */
    public String postPlainBody(String url, String content, Map<String, String> headerMap) {
        Request.Builder request = new Request.Builder()
                .url(url)
                .post(RequestBody.create(MediaType.parse(String.valueOf(ContentType.TEXT_PLAIN)), content));

        return extractResponse(request, headerMap);
    }

    /**
     * 提交json格式的body请求
     * @param url     请求地址
     * @param content 请求的json格式body
     * @return 响应内容
     */
    public String postJsonBody(String url, String content, Map<String, String> headerMap) {
        Request.Builder request = new Request.Builder()
                .url(url)
                .post(RequestBody.create(MediaType.parse(String.valueOf(ContentType.APPLICATION_JSON)), content));
        return extractResponse(request, headerMap);
    }

    /**
     * 同步阻塞获取文件
     * @param url      请求地址
     * @param filePath 存储文件路径，如果存在目录，需要先创建目录
     */
    public void download(String url, String filePath) {
        Request request = new Request.Builder().url(url).build();

        OkHttpClient client = new OkHttpClient();

        try {
            Response response = client.newCall(request).execute();
            saveFile(response, url, filePath);
        } catch (Exception e) {
            log.error("文件下载失败，请求地址为：{}", url);
        }
    }

    private void saveFile(Response response, String url, String filePath) {
        InputStream is = null;
        FileOutputStream fos = null;

        try {
            if (response.body() == null) {
                log.error("URL：{}对应的文件不存在", url);
                return;
            }
            byte[] buf = new byte[2048];
            int len;
            // 储存下载文件的目录
            is = response.body().byteStream();
            File file = new File(filePath);
            FileUtils.checkAndCreateFile(file);

            fos = new FileOutputStream(file);
            while ((len = is.read(buf)) != -1) {
                fos.write(buf, 0, len);
            }
            fos.flush();
        } catch (Exception e) {
            this.logger.error(e.getMessage(), e);
        } finally {
            try {
                if (is != null) {
                    is.close();
                }
            } catch (IOException e) {
                //
            }
            try {
                if (fos != null) {
                    fos.close();
                }
            } catch (IOException e) {
                //
            }
        }
    }

    /**
     * 异步下载文件，无法立即获取文件
     * @param url      下载路径
     * @param filePath 文件路径
     */
    public void downloadAysn(String url, String filePath) {
        Request request = new Request.Builder().url(url).build();

        OkHttpClient client = new OkHttpClient();
        client.newCall(request).enqueue(new Callback() {
            @Override
            public void onFailure(Call call, IOException e) {
                log.error("文件下载失败，请求地址为：{}", url);
                log.error(e.getMessage(), e);
            }

            @Override
            public void onResponse(Call call, Response response) {
                saveFile(response, url, filePath);
            }
        });
    }
}
