package com.bxm.newidea.component.jwt.tools;

import com.google.common.collect.Maps;
import io.jsonwebtoken.Claims;
import io.jsonwebtoken.Jwts;
import io.jsonwebtoken.SignatureAlgorithm;

import java.util.Date;
import java.util.Map;

import static com.bxm.newidea.component.jwt.constant.JwtContant.CLAIM_KEY_CREATED;
import static com.bxm.newidea.component.jwt.constant.JwtContant.CLAIM_KEY_KEY;
import static com.bxm.newidea.component.jwt.constant.JwtContant.TOKEN_SECRET;

/**
 * jwt token操作工具
 */
public class JwtTokenUtil {

    private JwtTokenUtil() {
    }

    /**
     * 解析token，从token中获取用户ID
     * @param token 用户当前持有的token
     * @return token中包含的用户ID
     */
    public static String parseToken(String token) {
        String username;
        try {
            final Claims claims = getClaimsFromToken(token);
            username = claims.getSubject();
        } catch (Exception e) {
            username = null;
        }
        return username;
    }

    private static Date getCreatedDateFromToken(String token) {
        Date created;
        try {
            final Claims claims = getClaimsFromToken(token);
            created = new Date((Long) claims.get(CLAIM_KEY_CREATED));
        } catch (Exception e) {
            created = null;
        }
        return created;
    }

    private static Date getExpirationDateFromToken(String token) {
        Date expiration;
        try {
            final Claims claims = getClaimsFromToken(token);
            expiration = claims.getExpiration();
        } catch (Exception e) {
            expiration = null;
        }
        return expiration;
    }

    private static Claims getClaimsFromToken(String token) {
        Claims claims;
        try {
            claims = Jwts.parser()
                    .setSigningKey(TOKEN_SECRET)
                    .parseClaimsJws(token)
                    .getBody();
        } catch (Exception e) {
            claims = null;
        }
        return claims;
    }

    private static Boolean isCreatedBeforeLastPasswordReset(Date created, Date lastPasswordReset) {
        return (lastPasswordReset != null && created.before(lastPasswordReset));
    }

    /**
     * 根据用户ID创建token
     * @param userId 用户ID
     * @return 创建的token值
     */
    public static String generateToken(Long userId) {
        Map<String, Object> claims = Maps.newHashMap();
        claims.put(CLAIM_KEY_KEY, userId);
        claims.put(CLAIM_KEY_CREATED, new Date());
        return generateToken(claims);
    }

    private static String generateToken(Map<String, Object> claims) {
        return Jwts.builder()
                .setClaims(claims)
                .setExpiration(generateExpirationDate())
                .signWith(SignatureAlgorithm.HS512, TOKEN_SECRET)
                .compact();
    }

    /**
     * token是否可被刷新
     * @param token     当前token
     * @param lastReset 最后重置时间
     * @return true表示可刷新
     */
    public static Boolean canTokenBeRefreshed(String token, Date lastReset) {
        final Date created = getCreatedDateFromToken(token);
        return !isCreatedBeforeLastPasswordReset(created, lastReset) && !isTokenExpired(token);
    }

    /**
     * 根据颁发的refreshToken刷新accessToken
     * @param token accessToken
     * @return 刷新后的新accessToken
     */
    public static String refreshToken(String token) {
        String refreshedToken;
        try {
            final Claims claims = getClaimsFromToken(token);
            claims.put(CLAIM_KEY_CREATED, new Date());
            refreshedToken = generateToken(claims);
        } catch (Exception e) {
            refreshedToken = null;
        }
        return refreshedToken;
    }

    private static Boolean isTokenExpired(String token) {
        final Date expiration = getExpirationDateFromToken(token);
        return expiration.before(new Date());
    }

    public static Date generateExpirationDate() {
        long expiration = 72 * 24 * 60 * 60 * 1000L;
        return new Date(System.currentTimeMillis() + expiration);
    }
}