package com.bxm.component.mybatis.dbshunt.strategy.handle;

import com.bxm.component.mybatis.dbshunt.entity.ShuntTableLog;
import com.bxm.component.mybatis.dbshunt.enums.DBShuntTypeEnum;
import com.bxm.component.mybatis.dbshunt.param.ShuntParam;
import com.bxm.newidea.component.JSON;
import com.bxm.newidea.component.uuid.SequenceCreater;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.time.DateUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;
import java.util.List;
import java.util.Objects;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * 处理近多少天的数据
 *
 * @author lowi
 * @date 2021/6/28 16:50
 */
@Component
@Slf4j
public class InitTableNameByDayHandle extends AbstractTableHandler {

    @Autowired
    private SequenceCreater sequenceCreater;


    public static final String DATE_DAY_FORMAT = "yyyyMMdd";

    public static ThreadLocal<DateFormat> PATTERN_DAY_FORMAT = ThreadLocal.withInitial(() -> new SimpleDateFormat(DATE_DAY_FORMAT));


    @Override
    public void tableDataHandler(ShuntParam shuntParam) {

        if (shuntParam.getTypeValue() != null && shuntParam.getTypeValue() <= 0) {
            log.error("规则类型不能<=0，shuntParam：{}", JSON.toJSONString(shuntParam));
            return;
        }

        //初始化分表数据
        initShuntTableData(shuntParam.getDataName());

        //查询最后一条记录
        ShuntTableLog shuntTableLog = getLastLogByTableNameAndType(shuntParam.getTableName(), shuntParam.getDbShuntTypeEnum());

        //不存在信息，则生成新的分表信息
        if (shuntTableLog == null) {
            shuntTableLog = insertTableLog(shuntParam);
        } else {
            shuntTableLog = judgeTableLogInfoExpire(shuntTableLog, shuntParam);
        }

        //最终入库操作
        finalHandlerData(shuntTableLog, shuntParam);

    }

    /**
     * 最终入库操作
     *
     * @param shuntTableLog 表信息
     */
    private void finalHandlerData(ShuntTableLog shuntTableLog, ShuntParam shuntParam) {
        List<String> tableNameList = getTableNameList(shuntParam.getDataName(), shuntParam.getTableName());

        if (tableNameList.isEmpty()) {
            log.error("数据库表不存在");
            return;
        }

        //新的表名已 老表名+按维度分的类型+时间后缀
        String newTableName = shuntTableLog.getTableName() + "_" + shuntTableLog.getType() + "_" + shuntTableLog.getSuffix();
        //按近多少天分 时间只要小于后缀最后一天即可
        String dateStr = getDateStr(shuntParam.getTypeValue());

        //创建新表操作
        initNewNameTable(newTableName, shuntParam);

        boolean isHandler = false;
        //则说明表做了分表，或者前缀结构相似的比较多
        if (tableNameList.size() > 1) {
            //正则结构为表名为t_user 信息为 t_user_0 或者 t_user_10
            Pattern pattern = Pattern.compile(shuntParam.getTableName() + "_[\\d]{1,}$");
            for (String tableName : tableNameList) {
                Matcher matcher = pattern.matcher(tableName);

                //正则匹配到了，才做操作
                if (matcher.find()) {
                    insertAndDelData(tableName, newTableName, shuntParam.getConditionField(), dateStr);
                    isHandler = true;
                }
            }
        } else {
            insertAndDelData(tableNameList.get(0), newTableName, shuntParam.getConditionField(), dateStr);
            isHandler = true;
        }

        //如果此标识为false则说明没操作，t_user没做分表 而是查询到了 此操作导致生成的新表t_user_1_20210614
        //所以需要特殊处理
        if (!isHandler) {
            insertAndDelData(tableNameList.get(0), newTableName, shuntParam.getConditionField(), dateStr);
        }
    }

    /**
     * 获取时间条件的数据
     *
     * @return 时间字符串
     */
    private String getDateStr(Integer dayNums) {
        //近shuntParam.getTypeValue()日的时间计算 当前时间-shuntParam.getTypeValue() ，计算出时间
        Calendar calendar = Calendar.getInstance();
        calendar.add(Calendar.DAY_OF_MONTH, dayNums * -1);
        calendar.set(Calendar.HOUR_OF_DAY, 23);
        calendar.set(Calendar.MINUTE, 59);
        calendar.set(Calendar.SECOND, 59);

        Date calendarTime = calendar.getTime();
        return com.bxm.newidea.component.tools.DateUtils.formatDateTime(calendarTime);
    }


    private void insertAndDelData(String oldTableName, String newTableName, String conditionField, String dateStr) {

        while (true) {

            int existDataByDate = dbShuntMapper.isExistDataByDate(oldTableName, conditionField, dateStr);

            //有数据就一直执行
            //没数据了 跳出
            if (existDataByDate == 0) {
                break;
            }

            dbShuntMapper.insertDataToNewTable(newTableName, oldTableName, conditionField, dateStr);

            dbShuntMapper.deleteOldTableData(oldTableName, conditionField, dateStr);
        }
    }


    /**
     * 插入表数据
     *
     * @param shuntParam 参数
     * @return 返回信息
     */
    private ShuntTableLog insertTableLog(ShuntParam shuntParam) {

        ShuntTableLog shuntTableLog = new ShuntTableLog();
        shuntTableLog.setSuffix(tableSuffixHandler(shuntParam));
        shuntTableLog.setId(sequenceCreater.nextLongId());
        shuntTableLog.setCreateTime(new Date());
        shuntTableLog.setTableName(shuntParam.getTableName());
        shuntTableLog.setType(shuntParam.getDbShuntTypeEnum().getType());
        shuntTableLog.setTypeValue(shuntParam.getTypeValue());

        dbShuntMapper.insertTableNameLog(TABLE_LOG_NAME, shuntTableLog);
        return shuntTableLog;
    }

    /**
     * 表后缀处理
     *
     * @return 返回表的后缀
     */
    private String tableSuffixHandler(ShuntParam shuntParam) {
        Date date = DateUtils.addDays(new Date(), shuntParam.getTypeValue() * -1);
        return PATTERN_DAY_FORMAT.get().format(date);
    }


    /**
     * 判断表中的信息是否已过期（该插入新数据了）
     *
     * @param shuntTableLog 原表信息
     * @param shuntParam    入参
     * @return 数据
     */
    private ShuntTableLog judgeTableLogInfoExpire(ShuntTableLog shuntTableLog, ShuntParam shuntParam) {

        //如果最新一条的数据 和 入参值不同，则说明改变了近多少天的策略，需要重新生成一天数据
        if (!Objects.equals(shuntTableLog.getTypeValue(), shuntParam.getTypeValue())) {
            return insertTableLog(shuntParam);
        }

        Date insertTime = shuntTableLog.getCreateTime();
        Calendar cal = Calendar.getInstance();
        cal.setTime(insertTime);
        cal.set(Calendar.HOUR_OF_DAY, 0);
        cal.set(Calendar.MINUTE, 0);
        cal.set(Calendar.SECOND, 0);
        cal.set(Calendar.MILLISECOND, 0);
        Date insertDate = cal.getTime();

        //近shuntParam.getTypeValue()日的时间计算 当前时间-shuntParam.getTypeValue() ，计算出时间
        Calendar calendar = Calendar.getInstance();
        calendar.add(Calendar.DAY_OF_MONTH, shuntParam.getTypeValue() * -1);
        calendar.set(Calendar.HOUR_OF_DAY, 0);
        calendar.set(Calendar.MINUTE, 0);
        calendar.set(Calendar.SECOND, 0);
        calendar.set(Calendar.MILLISECOND, 0);
        Date calendarTime = calendar.getTime();

        //如果当前时间为2021-06-30  shuntParam.getTypeValue()  = 30
        //计算时间为 2021-05-30
        //如果计算出的时间为 已经>= 插入的时间，说明表中的数据已经
        // 2021-05-30 > 2021-05-29,说明此分表的数据已经存储了30天的，需要建 2021630的新表了
        //这张表中已经记录了 2021-5-29 -- 2021-6-29的一个月数据
        if (calendarTime.getTime() >= insertDate.getTime()) {
            return insertTableLog(shuntParam);
        }
        return shuntTableLog;
    }


    @Override
    public DBShuntTypeEnum getType() {
        return DBShuntTypeEnum.DAY;
    }
}
