package com.bxm.component.httpclient.config;

import lombok.extern.slf4j.Slf4j;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.conn.HttpClientConnectionManager;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.impl.conn.PoolingHttpClientConnectionManager;
import org.springframework.boot.autoconfigure.AutoConfigureAfter;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

/**
 * httpClient客户端配置
 * 防止和feign中的配置冲突，优先使用feign中配置的httpClient
 *
 * @author liujia
 */
@Configuration
@Slf4j
@ConditionalOnMissingBean(HttpClientConnectionManager.class)
@EnableConfigurationProperties(HttpClientProperties.class)
@AutoConfigureAfter(name = "org.springframework.cloud.openfeign.FeignAutoConfiguration")
public class HttpClientAutoConfiguration {

    private HttpClientProperties httpClientProperties;

    public HttpClientAutoConfiguration(HttpClientProperties httpClientProperties) {
        this.httpClientProperties = httpClientProperties;
    }

    /**
     * 使用httpclient连接池管理器
     */
    private HttpClientConnectionManager httpClientConnectionManager() {
        PoolingHttpClientConnectionManager poolingHttpClientConnectionManager = new PoolingHttpClientConnectionManager();
        //连接池最大连接数
        poolingHttpClientConnectionManager.setMaxTotal(httpClientProperties.getMaxTotal());
        //设置每个主机地址的最大并发数
        poolingHttpClientConnectionManager.setDefaultMaxPerRoute(50);
        return poolingHttpClientConnectionManager;
    }

    /**
     * 配置连接参数
     */
    @Bean
    @ConditionalOnMissingBean(RequestConfig.class)
    public RequestConfig requestConfig() {
        log.debug("使用了component-net中配置的http连接参数");

        //连接配置
        return RequestConfig.custom()
                //创建连接的最长时间
                .setConnectTimeout(httpClientProperties.getConnectionTimeout())
                //从连接池中获取到连接的最长时间
                .setConnectionRequestTimeout(httpClientProperties.getWaitTimeout())
                //数据传输的最长时间
                .setSocketTimeout(httpClientProperties.getReadTimeout())
                .build();
    }

    /**
     * 注册httpClient客户端
     */
    @Bean
    @ConditionalOnMissingBean(CloseableHttpClient.class)
    public CloseableHttpClient httpClient() {
        //配置httpclientBuilder
        HttpClientBuilder httpClientBuilder = HttpClients.custom()
                //设置连接池
                .setConnectionManager(httpClientConnectionManager())
                //设置连接的参数
                .setDefaultRequestConfig(requestConfig());
        return httpClientBuilder.build();
    }

}
