package com.bxm.component.office.excel.export.impl;

import com.bxm.component.office.excel.config.ExcelConstant;
import com.bxm.component.office.excel.export.ExcelExporter;
import com.bxm.component.office.excel.export.setting.ExcelSheet;
import com.bxm.component.office.excel.export.setting.ExportSetting;
import com.bxm.component.office.excel.export.setting.SheetHeader;
import com.bxm.component.office.excel.export.themes.ExcelThemesManage;
import com.bxm.component.office.excel.format.CellValueConvertContext;
import com.bxm.component.office.excel.format.CellValueConverter;
import com.bxm.component.office.excel.format.CellValueManage;
import com.bxm.component.office.excel.format.config.CellTypeEnum;
import com.bxm.component.office.exception.ExportException;
import org.apache.commons.io.FileUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.poi.hssf.usermodel.HSSFCell;
import org.apache.poi.hssf.usermodel.HSSFRow;
import org.apache.poi.hssf.usermodel.HSSFSheet;
import org.apache.poi.hssf.usermodel.HSSFWorkbook;
import org.apache.poi.ss.usermodel.CellStyle;
import org.apache.poi.ss.util.CellRangeAddress;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.util.CollectionUtils;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.OutputStream;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * 功能说明：<br/>
 * Excel导出接口的抽象实现，包含excel导出的公共功能
 * <br/>
 * 开发时间：2016年3月3日<br/>
 */
public abstract class AbstractExcelExport implements ExcelExporter {

    protected Logger LOGGER = LoggerFactory.getLogger(getClass());

    protected ExcelThemesManage themesManage;

    protected CellValueManage cellValueManage;

    /**
     * 当前操作的workbook
     */
    protected HSSFWorkbook workbook;

    /**
     * 当前操作的sheet
     */
    protected HSSFSheet currentSheet = null;

    /**
     * 当前操作的行对象
     */
    protected HSSFRow currentRow = null;

    /**
     * 当前操作的列对象
     */
    protected HSSFCell currentCell = null;

    /**
     * 导出配置参数
     */
    protected ExportSetting setting;

    /**
     * 导出文件
     */
    protected File excelFile;

    /**
     * 输出文件流
     */
    protected OutputStream excelFileOutputStream;

    /**
     * 当前操作行数
     */
    protected int currentRowNum = 0;

    /**
     * 当前操作列数
     */
    protected int currentColNum = 0;

    /**
     * 表格标题样式
     */
    private CellStyle titleCellStyle;

    /**
     * 表格头样式
     */
    private CellStyle headCellStyle;

    /**
     * 表格体样式
     */
    private CellStyle contentCellStyle;

    /**
     * 当前是否是第一行
     */
    private boolean isNotFirstRow = false;

    /**
     * 跨行影响列记录
     * Map<行数,<列数,该列跨行数>>
     */
    private Map<Integer, Map<Integer, Integer>> rowAffect = new HashMap<Integer, Map<Integer, Integer>>();

    @Override
    public void export(OutputStream excelFileStream, ExportSetting setting) {
        this.excelFileOutputStream = excelFileStream;
        this.setting = setting;

        commonExport();
    }

    private void commonExport() {
        // 参数校验
        if (assertExportSetting()) {
            LOGGER.error("excel导出所需参数不全");
            return;
        }
        getContext().addExtend(ExcelConstant.EXPORT_SETTING, setting);
        try {
            // 构建workbook
            createWorkbook();
            // 用户自定义excel操作
            execExport();
            write();
        } catch (Exception e) {
            LOGGER.error(e.getMessage(), e);
            throw new ExportException(e);
        }
    }

    @Override
    public void export(File excelFile, ExportSetting setting) {
        this.excelFile = excelFile;
        this.setting = setting;

        commonExport();
    }

    /**
     * excel导出excel所需参数校验
     * @return 是否通过校验，true为校验失败
     */
    private boolean assertExportSetting() {
        // 如果未传setting，则直接通过校验（兼容之前版本代码）
        if (setting == null) {
            LOGGER.warn("ExportSetting应为必须参数，为了兼容老版本代码允许为null，此处可能会出现潜在错误");
            return false;
        }
        // 通用校验
        if (CollectionUtils.isEmpty(setting.getSheets())) {
            LOGGER.error("ExportSetting中未配置ExcelSheet");
            return true;
        }
        for (ExcelSheet excelSheet : setting.getSheets()) {// 遍历所有sheet页，如果sheet页中即没有header也没有扩展数据，则校验失败
            if (CollectionUtils.isEmpty(excelSheet.getHeaders()) && !excelSheet.hasExtendData()) {
                LOGGER.error("ExcelSheet中既没有配置SheetHeader，也没有配置扩展数据addExtendData");
                return true;
            }
        }

        // 若通过通用校验，执行自定义校验
        return assertSetting();
    }

    /**
     * 执行自定义的excel操作后导出excel。
     */
    protected abstract void execExport();

    protected CellValueConvertContext getContext() {
        return this.cellValueManage.getContext();
    }

    /**
     * 校验excel文件是否符合需求<br/>
     * <li>文件不能为空
     * <li>文件必须是文件，而不是文件夹
     * @param excelFile excel文件
     * @return 是否校验成功, true表示校验失败
     */
    protected boolean assertFile(File excelFile) {
        boolean result = null == excelFile;

        if (result) {
            LOGGER.error("导出Excel文件不是一个有效的导出文件");
        }

        return result;
    }

    /**
     * 对导出参数进行校验，若需自定义校验重写即可
     * @return true表示校验失败
     */
    protected boolean assertSetting() {
        return false;
    }

    /**
     * 构建HssfWorkbook对象
     */
    protected void createWorkbook() {
        workbook = new HSSFWorkbook();
        getContext().setWorkbook(workbook);
        // 初始化样式管理器
        themesManage.init(workbook);
        // 初始化表格样式
        initStyle();
    }

    /**
     * 将构建的workbook写入指定的文件
     * @throws IOException
     */
    protected void write() throws IOException {
        if (excelFileOutputStream != null) {
            workbook.write(excelFileOutputStream);
            excelFileOutputStream.close();
            return;
        }
        if (excelFile != null) {
            ByteArrayOutputStream os = new ByteArrayOutputStream();
            workbook.write(os);

            byte[] fileContent = os.toByteArray();
            FileUtils.writeByteArrayToFile(excelFile, fileContent);
            os.close();
        }
    }

    /**
     * 初始化表格标题，表格头，表格体样式，样式为样式管理器提供的默认样式。<br>
     * 如需自定义表格标题、表格头、表格体样式则可以自己赋值。<br>
     * 如需自定义某个单元格样式可自己创建样式并在创建单元格时将样式加入。
     * 样式创建的数量有限，不能创建过多，相同的样式需要进行复用
     */
    private void initStyle() {
        // 表格标题
        titleCellStyle = themesManage.getTitleStyle();
        // 表头样式
        headCellStyle = themesManage.getHeadStyle();
        // 表格体样式
        contentCellStyle = themesManage.getContentStyle();
    }

    /**
     * 创建一个新的sheet页
     */
    protected final void newSheet() {
        newSheet("");
    }

    /**
     * 创建一个新的sheet页
     * @param sheet sheet页配置参数
     */
    protected final void newSheet(ExcelSheet sheet) {
        newSheet(sheet.getSheetName());
        setFreezePanel(sheet);
        setSheetColumnWidth(sheet);
    }

    /**
     * 创建一个新的sheet页，并设置sheet页名称<br>
     * <b>ps</b>:sheet页名不可以包含特殊字符<br>
     */
    protected final void newSheet(String sheetName) {

        // 初始化行数
        currentRowNum = 0;
        currentColNum = 0;
        isNotFirstRow = false;

        if (StringUtils.isBlank(sheetName)) {
            currentSheet = workbook.createSheet();
        } else {
            currentSheet = workbook.createSheet(sheetName);
        }
        getContext().setSheet(currentSheet);
    }

    /**
     * 创建一行
     */
    protected final void newRow() {
        newRow(1);
    }

    /**
     * 跳过rowNum行后创建一行
     * @param rowNum 创建的行数，必须为<b>正数</b>
     */
    protected final void newRow(int rowNum) {
        if (getContext().getSheet() == null) {
            throw new ExportException("请先创建sheet页(newSheet()方法)，再执行对sheet页的设置等操作");
        }
        if (rowNum < 1) {
            throw new ExportException("调用newRow(int rowNum)时异常，参数：{rowNum：[" + rowNum + "]}中创建的行数rowNum必须>0");
        }

        if (isNotFirstRow) {// 若不是第一行
            // 当前行数+1
            currentRowNum += rowNum;
        }
        // 当前列数置为0
        currentColNum = 0;

        currentRow = currentSheet.createRow(currentRowNum);
        getContext().setRow(currentRow);

        if (currentRowNum == 0) {
            isNotFirstRow = true;
        }
    }

    /**
     * 设置表格体中每个cell中的内容，跨行列，单元格默认String类型<br>
     * <b>本方法不可重写，所有参数均不可为空，若扩展请注意，如参数不需较多，可以使用其他重载方法，方法如下</b><br>
     * <li>setCellContent(args..)</li>
     * <li>setHeadCellContent(args..)</li>
     * <li>setContentCellContent(args..)</li>
     * @param content     单元格内容内容，该内容需要为单元格数据类型转换器 {@link CellValueConverter} 支持的类型
     * @param cellType    设置单元格数据类型
     * @param rowspan     行数 必须大于0
     * @param colspan     列数 必须大于0
     * @param customStyle 自定义样式
     */
    protected final void setCellContent(Object content, CellTypeEnum cellType, Integer rowspan, Integer colspan,
                                        CellStyle customStyle) {

        if (getContext().getSheet() == null) {
            throw new ExportException("请先创建sheet页(newSheet())，再执行对sheet页的设置等操作");
        }
        if (rowspan <= 0 || colspan <= 0) {
            throw new ExportException("调用单元格设值函数时异常，参数：{rowspan：[" + rowspan + "]，colspan：[" + colspan + "]}中设置的行、列数必须>0");
        }
        if (cellType == null || customStyle == null) {
            throw new ExportException("调用单元格设值函数时异常，参数：{content：[" + content + "]，cellType[" + cellType + "]，customStyle：["
                    + customStyle + "]}中存在空参数，参数不能为空");
        }

        // 设置跨行影响的列宽度
        setRowspan(rowspan, colspan);

        // 处理跨行
        dealRowspan(cellType, customStyle);

        // 设置cell的值
        setCellValue(cellType, content, customStyle);

        // 合并处理
        dealMerge(rowspan, colspan, customStyle);

        currentColNum += colspan;
    }

    /**
     * 给cell进行设值，并将cell添加到上下文中
     * @param cellType  单元格数据类型
     * @param value     单元格值（该值需要是excel单元格类型转换器支持的数据格式，若不支持请扩展类型转换器）
     * @param cellStyle 单元格样式
     */
    private void setCellValue(CellTypeEnum cellType, Object value, CellStyle cellStyle) {

        currentCell = currentRow.createCell(currentColNum);
        currentCell.setCellStyle(cellStyle);

        getContext().setCell(currentCell);

        if (null != cellType) {

            getContext().setCellType(cellType);
        }

        getContext().addExtend(ExcelConstant.SET_CELL_VALUE, value);

        cellValueManage.setValue();
    }

    /**
     * 处理合并单元格
     * @param rowspan   行数
     * @param colspan   列数
     * @param cellStyle 单元格样式
     */
    private void dealMerge(Integer rowspan, Integer colspan, CellStyle cellStyle) {
        CellRangeAddress rangeAddress = new CellRangeAddress(currentRowNum, currentRowNum + rowspan - 1, currentColNum,
                currentColNum + colspan - 1);
        currentSheet.addMergedRegion(rangeAddress);

        for (int i = 1; i < colspan; i++) {

            currentCell = currentRow.createCell(currentColNum + i);
            currentCell.setCellStyle(cellStyle);
        }
    }

    /**
     * 设置跨行的信息，在处理跨行时根据该数据进行处理
     * @param rowspan 行数
     * @param colspan 列数
     */
    private void setRowspan(Integer rowspan, Integer colspan) {

        if (rowspan > 1) {
            // 遍历跨的行数
            for (int i = 1; i < rowspan; i++) {
                // 获取各行受影响的列数
                Map<Integer, Integer> cols = rowAffect.get(currentRowNum + i);
                if (CollectionUtils.isEmpty(cols)) {
                    cols = new HashMap<Integer, Integer>();
                }
                // 设置当前行影响的列数
                cols.put(currentColNum, colspan);
                rowAffect.put(currentRowNum + i, cols);
            }
        }
    }

    /**
     * 处理跨行，根据设置跨行的位置进行创建空值cell
     * @param cellType  单元格数据类型
     * @param cellStyle 样式
     */
    private void dealRowspan(CellTypeEnum cellType, CellStyle cellStyle) {
        Map<Integer, Integer> cols = rowAffect.get(currentRowNum);
        if (!CollectionUtils.isEmpty(cols) && null != cols.get(currentColNum) && 0 != cols.get(currentColNum)) {
            // 当前列受影响的列数
            int colsNum = cols.get(currentColNum);
            for (int i = 0; i < colsNum; i++) {
                setCellValue(cellType, "", cellStyle);
                this.currentColNum++;
            }
            // 跨行处理在设置单元格前进行，连续的跨行需要递归处理
            dealRowspan(cellType, cellStyle);
        }
    }

    /**
     * 从当前行创建一个指定列{@code colNum}的单元格并设定样式，用于修复合并单元格时最后的单元格的边框样式问题
     * @param colNum    列号
     * @param cellStyle 使用的样式
     */
    protected void fixStyle(int colNum, CellStyle cellStyle) {
        currentColNum = colNum;
        setCellValue(CellTypeEnum.STRING, "", cellStyle);
    }

    /**
     * 设置标题
     * @param title 标题（默认string类型标题，如果是复杂标题，请直接调用设置单元格方法生成）
     */
    protected final void setExcelTitle(String title, CellTypeEnum cellType, Integer rowspan, Integer colspan) {
        if (getContext().getSheet() == null) {
            throw new ExportException("请先创建sheet页(newSheet())，再执行对sheet页的设置等操作");
        }
        if (rowspan <= 0 || colspan <= 0) {
            throw new ExportException("调用设置函数setExcelTitle时异常，参数：{rowspan：[" + rowspan + "]，colspan：[" + colspan
                    + "]}中设置的行、列数必须>0");
        }
        if (title == null) {
            throw new ExportException("调用设置函数setExcelTitle时异常，参数：{title：[" + title + "]}中表格标题title为空，参数不能为空");
        }

        newRow();
        setCellContent(title, cellType, rowspan, colspan, titleCellStyle);
    }

    /**
     * 获取ExportSetting中设置的第index个sheet页，<b>index从1开始</b>
     * <b>注意：index从1开始</b>
     * @param index sheet页序号
     * @return
     */
    protected final ExcelSheet getSheet(int index) {
        if (index > setting.getSheets().size()) {
            throw new ExportException("调用getSheet(int index)函数异常，参数：{index：[" + index
                    + "]}中参数index越界，index不能超过controller中设置sheet页的个数");
        }
        return setting.getSheets().get(index - 1);
    }

    /**
     * 设置通用列宽，将当前sheet页的所有列宽改为设置的值
     * @param commonWidth 通用的列宽
     * @param countCol    要设置多少列
     */
    protected final void setSheetCommonColWidth(int commonWidth, int countCol) {
        if (getContext().getSheet() == null) {
            throw new ExportException("请先创建sheet页(newSheet())，再执行对sheet页的设置等操作");
        }
        if (commonWidth < 0 || countCol <= 0) {
            throw new ExportException("调用setSheetCommonColWidth函数异常，参数：{commonWidth：[" + commonWidth + "]，countCol：[" + countCol
                    + "]}中参数越界，请传递大于0的宽度和列数");
        }
        for (int i = 0; i < countCol; i++) {
            getContext().getSheet().setColumnWidth(i, commonWidth * 256);
        }
    }

    /**
     * 设置当前sheet页自定义列宽，设置第几列的列宽为多少，<b>列序号从1开始</b><br>
     * <b>注意：列序号从1开始</b><br>
     * @param colNum   第几列
     * @param colWidth 宽度是多少
     */
    protected final void setSheetCustomColWidth(int colNum, int colWidth) {
        if (colNum <= 0 || colWidth <= 0) {
            throw new ExportException("调用setSheetCommonColWidth函数异常，参数：{colNum：[" + colNum + "]，colWidth：[" + colWidth
                    + "]}中参数越界，请传递大于0的宽度和列数");
        }
        getContext().getSheet().setColumnWidth(colNum - 1, colWidth * 256);
    }

    /**
     * 设置窗口冻结区域（冻结前几行/前几列，行/列中不需要冻结的设置为0即可）
     * @param freezeRows 冻结前几行
     * @param freezeCols 冻结前几列
     */
    protected final void setFreezePane(int freezeRows, int freezeCols) {
        if (getContext().getSheet() == null) {
            throw new ExportException("请先创建sheet页(newSheet())，再执行对sheet页的设置等操作");
        }
        if (freezeRows < 0 || freezeCols < 0) {
            throw new ExportException("调用setFreezePane函数异常，参数：{freezeRows：[" + freezeRows + "]，freezeCols：[" + freezeCols
                    + "]}中参数越界，请传递大于等于0的行数和列数");
        }
        // 冻结前两行
        currentSheet.createFreezePane(freezeRows, freezeCols, freezeRows, freezeCols);
    }

    /**
     * 设置单元格的值（单元格数据类型为String类型）<br>
     * <b>自定义单元格行数，自定义单元格列数，自定义单元格样式</b>
     * @param content     单元格内容内容，该内容需要为单元格数据类型转换器 {@link CellValueConverter} 支持的类型
     * @param rowspan     行数
     * @param colspan     列数
     * @param customStyle 自定义样式
     */
    protected void setCellContent(Object content, Integer rowspan, Integer colspan, CellStyle customStyle) {
        setCellContent(content, CellTypeEnum.STRING, rowspan, colspan, customStyle);
    }

    /**
     * 设置单元格的值（1行1列）<br>
     * <b>自定义单元格数据类型，自定义单元样式</b>
     * @param content     单元格内容内容，该内容需要为单元格数据类型转换器 {@link CellValueConverter} 支持的类型
     * @param cellType    设置单元格数据类型
     * @param customStyle 自定义样式
     */
    protected void setCellContent(Object content, CellTypeEnum cellType, CellStyle customStyle) {
        setCellContent(content, cellType, 1, 1, customStyle);
    }

    /**
     * 设置单元格的值（1行1列）<br>
     * <b>自定义单元格数据类型，自定义单元样式</b>
     * @param content  单元格内容内容，该内容需要为单元格数据类型转换器 {@link CellValueConverter} 支持的类型
     * @param cellType 设置单元格数据类型
     */
    protected void setCellContent(Object content, CellTypeEnum cellType) {
        setCellContent(content, cellType, 1, 1, getContentCellStyle());
    }

    /**
     * 设置表格头单元格的值（自带表格头样式）<br>
     * <b>自定义单元格数据类型，自定义单元格行数，自定义单元格列数</b>
     * @param content  单元格内容内容，该内容需要为单元格数据类型转换器 {@link CellValueConverter} 支持的类型
     * @param cellType 设置单元格数据类型
     * @param rowspan  行数
     * @param colspan  列数
     */
    protected void setHeadCellContent(Object content, CellTypeEnum cellType, Integer rowspan, Integer colspan) {
        setCellContent(content, cellType, rowspan, colspan, getHeadCellStyle());
    }

    /**
     * 设置表格体单元格的值（自带表格体样式）<br>
     * <b>自定义单元格数据类型，自定义单元格行数，自定义单元格列数</b>
     * @param content  单元格内容内容，该内容需要为单元格数据类型转换器 {@link CellValueConverter} 支持的类型
     * @param cellType 设置单元格数据类型
     * @param rowspan  行数
     * @param colspan  列数
     */
    protected void setContentCellContent(Object content, CellTypeEnum cellType, Integer rowspan, Integer colspan) {
        setCellContent(content, cellType, rowspan, colspan, getContentCellStyle());
    }

    /**
     * 设置表格体单元格的值（单元格数据类型为String类型，1行1列）<br>
     * <b>自定义单元格样式</b>
     * @param content     单元格内容内容，该内容需要为单元格数据类型转换器 {@link CellValueConverter} 支持的类型
     * @param customStyle 单元格样式
     */
    protected void setCellContent(Object content, CellStyle customStyle) {
        setCellContent(content, CellTypeEnum.STRING, 1, 1, null == customStyle ? getContentCellStyle() : customStyle);
    }

    /**
     * 设置表格体单元格的值（单元格数据类型为String类型，1行1列）<br>
     * <b>自定义单元格样式</b>
     * @param content 单元格内容内容，该内容需要为单元格数据类型转换器 {@link CellValueConverter} 支持的类型
     */
    protected void setCellContent(Object content) {
        setCellContent(content, getContentCellStyle());
    }

    /**
     * 设置表格头单元格的值（自带表格头样式，单元格数据类型为String类型）<br>
     * <b>自定义单元格行数，自定义单元格列数</b>
     * @param content 单元格内容内容，该内容需要为单元格数据类型转换器 {@link CellValueConverter} 支持的类型
     * @param rowspan 行数
     * @param colspan 列数
     */
    protected void setHeadCellContent(Object content, Integer rowspan, Integer colspan) {
        setCellContent(content, CellTypeEnum.STRING, rowspan, colspan, getHeadCellStyle());
    }

    /**
     * 设置表格体单元格的值（自带表格体样式，单元格数据类型为String类型）<br>
     * <b>自定义单元格行数，自定义单元格列数</b>
     * @param content 单元格内容内容，该内容需要为单元格数据类型转换器 {@link CellValueConverter} 支持的类型
     * @param rowspan 行数
     * @param colspan 列数
     */
    protected void setContentCellContent(Object content, Integer rowspan, Integer colspan) {
        setCellContent(content, CellTypeEnum.STRING, rowspan, colspan, getContentCellStyle());
    }

    /**
     * 设置表格头单元格的值（自带表格头样式，1行1列）<br>
     * <b>自定义单元格数据类型</b>
     * @param content  单元格内容内容，该内容需要为单元格数据类型转换器 {@link CellValueConverter} 支持的类型
     * @param cellType 设置单元格数据类型
     */
    protected void setHeadCellContent(Object content, CellTypeEnum cellType) {
        setCellContent(content, cellType, 1, 1, getHeadCellStyle());
    }

    /**
     * 设置表格体单元格的值（自带表格体样式，1行1列）<br>
     * <b>自定义单元格样式</b>
     * @param content  单元格内容内容，该内容需要为单元格数据类型转换器 {@link CellValueConverter} 支持的类型
     * @param cellType 设置单元格数据类型
     */
    protected void setContentCellContent(Object content, CellTypeEnum cellType) {
        setCellContent(content, cellType, 1, 1, getContentCellStyle());
    }

    /**
     * 设置表格头单元格的值（仅设置值，单元格数据类型为String类型，1行1列，自带表格头样式）<br>
     * @param content 单元格内容内容，该内容需要为单元格数据类型转换器 {@link CellValueConverter} 支持的类型
     */
    protected void setHeadCellContent(Object content) {
        setCellContent(content, CellTypeEnum.STRING, 1, 1, getHeadCellStyle());
    }

    /**
     * 设置表格体单元格的值（单元格数据类型为String类型，1行1列，自带表格体样式）<br>
     * @param content 单元格内容内容，该内容需要为单元格数据类型转换器 {@link CellValueConverter} 支持的类型
     */
    protected void setContentCellContent(Object content) {
        setCellContent(content, CellTypeEnum.STRING, 1, 1, getContentCellStyle());
    }

    /**
     * 设置窗口冻结
     * @param sheet
     */
    protected void setFreezePanel(ExcelSheet sheet) {
        if (sheet.isFreezeHeader()) {
            int colIndex = 1;
            if (StringUtils.isNotBlank(sheet.getContentTitle())) {
                colIndex = 2;
            }
            // 冻结前两行
            currentSheet.createFreezePane(0, colIndex, 0, colIndex);
        }
    }

    /**
     * 设置当前sheet页的列宽
     * @param sheet
     */
    protected void setSheetColumnWidth(ExcelSheet sheet) {
        int defaultWidth = sheet.getDefaultWidth();

        for (SheetHeader header : sheet.getHeaders()) {
            if (0 == header.getWidth()) {
                if (0 == defaultWidth) {
                    currentSheet.autoSizeColumn(currentColNum);
                } else {
                    setSheetCustomColWidth(currentColNum + 1, defaultWidth);
                }
            } else {
                setSheetCustomColWidth(currentColNum + 1, header.getWidth());
            }

            currentColNum++;
        }
    }

    /**
     * 构建表头
     */
    protected void createHeader(List<SheetHeader> headers) {
        newRow();

        for (SheetHeader header : headers) {
            setCellContent(header.getTitle(), getHeadCellStyle());
        }
    }

    @Override
    public ExcelThemesManage getThemesManage() {
        return themesManage;
    }

    @Override
    @Autowired
    public void setThemesManage(ExcelThemesManage themesManage) {
        this.themesManage = themesManage;
    }

    @Override
    public CellValueManage getCellValueManage() {
        return cellValueManage;
    }

    @Override
    @Autowired
    public void setCellValueManage(CellValueManage cellValueManage) {
        this.cellValueManage = cellValueManage;
    }

    public CellStyle getTitleCellStyle() {
        return titleCellStyle;
    }

    public CellStyle getHeadCellStyle() {
        return headCellStyle;
    }

    public CellStyle getContentCellStyle() {
        return contentCellStyle;
    }

}
