package com.bxm.component.office.excel.format;

import com.bxm.component.office.excel.config.ExcelConstant;
import com.bxm.component.office.excel.format.impl.*;
import com.bxm.component.office.excel.parse.setting.ParseHeader;
import org.apache.poi.ss.usermodel.Cell;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.config.ConfigurableBeanFactory;
import org.springframework.context.annotation.Scope;
import org.springframework.core.Ordered;
import org.springframework.stereotype.Component;
import org.springframework.util.Assert;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;

/**
 * 功能说明：<br/>
 * 根据不同的单元格类型进行设值/取值的管理类,可根据情况进行转换器的扩展
 * <br/>
 * 开发时间：2016年3月3日<br/>
 */
@Component
@Scope(ConfigurableBeanFactory.SCOPE_PROTOTYPE)
public class CellValueManage implements InitializingBean {

    private List<CellValueConverter> converters = new ArrayList<>();

    private CellValueConvertContext context = new CellValueConvertContext();

    public CellValueConvertContext getContext() {
        return context;
    }

    public void setValue() {
        if (context.getCell() == null) {
            return;
        }

        for (CellValueConverter converter : converters) {
            if (converter.isMatch(context.getCellType())) {
                converter.setValue();
                break;
            }
        }
    }

    public Object getValue() {
        if (null == context.getCell()) {
            return null;
        }
        Assert.notNull(context.getCellType(), "列类型不能为空");

        Object value = null;

        // 单元格类型不为空时才进行处理
        if (context.getCell().getCellType() != Cell.CELL_TYPE_BLANK) {
            for (CellValueConverter converter : converters) {
                if (converter.isMatch(context.getCellType())) {
                    value = converter.getValue();
                    if (null == value) {
                        continue;
                    }
                    break;
                }
            }
        }

        // 未取到值时，尝试获取默认值
        ParseHeader header = (ParseHeader) context.getExtend(ExcelConstant.PARSE_HEADER);
        if (null == value && null != header && null != header.getDefaultValue()) {
            value = header.getDefaultValue().getValue();
        }

        return value;
    }

    /**
     * 添加自定义转换器
     * @param converter 自定义转换器
     */
    public void addConverter(CellValueConverter converter) {
        registerConverter(converter);
        sort();
    }

    /**
     * 将转换器按照优先级进行排序
     */
    private void sort() {
        converters.sort(Comparator.comparingInt(Ordered::getOrder));
    }

    private void registerConverter(CellValueConverter converter) {
        converter.setCellValueManage(this);
        converters.add(converter);
    }

    /**
     * 初始化默认的转换器
     */
    @Override
    public void afterPropertiesSet() {
        registerConverter(new StringCellConverter());
        registerConverter(new DateCellConverter());
        registerConverter(new TimestampCellConverter());
        registerConverter(new MoneyCellConverter());
        registerConverter(new NumericCellConverter());
        registerConverter(new TextCellConverter());
        registerConverter(new DefaultCellConverter());
        registerConverter(new BooleanCellConverter());

        sort();
    }

}
