package com.bxm.component.office.excel.parse.impl;

import com.bxm.component.office.excel.config.ExcelConstant;
import com.bxm.component.office.excel.format.CellValueConvertContext;
import com.bxm.component.office.excel.format.CellValueManage;
import com.bxm.component.office.excel.parse.ExcelParser;
import com.bxm.component.office.excel.parse.setting.ParseHeader;
import com.bxm.component.office.excel.parse.setting.ParseSetting;
import com.bxm.component.office.exception.ParseException;
import org.apache.commons.io.FileUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.poi.hssf.usermodel.HSSFWorkbook;
import org.apache.poi.ss.usermodel.Sheet;
import org.apache.poi.ss.usermodel.Workbook;
import org.apache.poi.xssf.usermodel.XSSFWorkbook;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.util.List;

/**
 * 功能说明：<br/>
 * excel文件解析类抽象类 <br/>
 * 开发时间：2016年4月22日<br/>
 */
public abstract class AbstractExcelParser<T> implements ExcelParser<T> {

    protected Logger LOGGER = LoggerFactory.getLogger(getClass());

    protected Sheet sheet;

    protected ParseSetting setting;

    protected CellValueManage cellValueManage;

    private Workbook workbook;

    private File excelFile;

    private InputStream excelInputStream;

    @Override
    public CellValueManage getCellValueManage() {
        return cellValueManage;
    }

    @Override
    @Autowired
    public void setCellValueManage(CellValueManage cellValueManage) {
        this.cellValueManage = cellValueManage;
    }

    @Override
    public List<T> parse(File excelFile, ParseSetting setting) {
        this.setting = setting;
        this.excelFile = excelFile;
        init();
        return parseSheet();
    }

    @Override
    public List<T> parse(InputStream inputStream, ParseSetting setting) {
        this.setting = setting;
        this.excelInputStream = inputStream;
        init();
        return parseSheet();
    }

    private void init() {
        assertSetting();
        try {
            workbook = createWorkbook();
            sheet = workbook.getSheetAt(0);

            getContext().setWorkbook(workbook);
            getContext().setSheet(sheet);
            getContext().addExtend(ExcelConstant.PARSE_SETTING, setting);
        } catch (IOException e) {
            LOGGER.error(e.getMessage(), e);
            throw new ParseException(e);
        }
    }

    protected abstract List<T> parseSheet();

    private void assertSetting() {
        if (null == setting) {
            throw new IllegalArgumentException("setting can not be null");
        }
        if (null == excelFile && null == excelInputStream) {
            throw new IllegalArgumentException("excelFile can not be null");
        }
        if (null == setting.getEntityClasz()) {
            throw new IllegalArgumentException("entityClasz can not be null");
        }
        if (null == setting.getHeads()) {
            throw new IllegalArgumentException("setting.heads can not be null");
        }
        for (ParseHeader header : setting.getHeads()) {
            if (!setting.isHeaderModel()) {
                if (header.getColIndex() < 0) {
                    throw new IllegalArgumentException("in the index model,colIndex must be greater than zero");
                }
            } else {
                if (StringUtils.isBlank(header.getTitle())) {
                    throw new IllegalArgumentException("in the title model,title must has text");
                }
            }
        }
    }

    protected CellValueConvertContext getContext() {
        return this.cellValueManage.getContext();
    }

    /**
     * 是否是Excel2003
     * @return 是否为2003
     */
    private boolean isExcel2003() {
        return StringUtils.endsWith(excelFile.getName(), ".xls");
    }

    protected boolean isExcel() {
        return excelFile != null && (StringUtils.endsWith(excelFile.getName(), ".xls")
                || StringUtils.endsWith(excelFile.getName(), ".xlsx"));
    }

    private Workbook createWorkbook() throws IOException {
        InputStream is = null;
        if (null != excelInputStream) {
            is = excelInputStream;
        } else {
            is = FileUtils.openInputStream(excelFile);
        }
        Workbook workbook = null;
        // 判断文件是否为Excel文件
        if (isExcel()) {
            if (isExcel2003()) {
                workbook = new HSSFWorkbook(is);
            } else {
                workbook = new XSSFWorkbook(is);
            }
        } else {
            // 文件不是一个标准的excel文件时，尝试进行构建，对构建过程中的异常不予处理
            try {
                // 尝试构建2003
                workbook = new HSSFWorkbook(is);
            } finally {
                // 如果构建2003版本失败，则尝试构建2007版本
                if (null == workbook) {
                    try {
                        workbook = new XSSFWorkbook(is);
                    } catch (Exception e) {
                        LOGGER.warn("输入的文件格式或流不是一个有效的Excel文件");
                    }
                }
            }
        }

        return workbook;
    }

}
