package com.bxm.component.oncejob.config;

import com.bxm.component.oncejob.enums.RepositoryStrategy;
import lombok.Data;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.context.properties.ConfigurationProperties;

/**
 * 一次性定时任务相关的配置参数
 *
 * @author liujia
 * @date 7/29/21 5:26 PM
 **/
@ConfigurationProperties(prefix = "component.once-job")
@Data
public class ComponentOnceJobConfigurationProperties {

    /**
     * 是否启用一次性定时任务
     */
    private boolean enable = true;

    /**
     * 拉取长期任务的时间点
     * 不支持变更，影响了任务分片的计算方式
     */
    private final int fetchLongTermJobMills = 30 * 60 * 1000;

    /**
     * 拉取距离当前多久之后的任务
     * 不支持变更，影响了任务分片的计算方式
     */
    private final int fetchIntervalMills = 5 * 1000;

    /**
     * 定期清理脏数据
     */
    private int clearDirtyIntervalMills = 30 * 1000;

    /**
     * 每次最多拉取的数据量
     */
    private int maxFetchNum = 100;

    /**
     * 从长期存储中拉取任务的每页数量
     */
    private int longTermFetchPageSize = 1000;

    /**
     * 任务单次执行花费的时间，超过此时间，将会记录waring日志
     */
    private long jobExpendWarningMills = 200L;

    /**
     * 当前应用的名称
     * 如未配置则将获取当前服务的默认服务名
     */
    @Value("${spring.application.name:undefined}")
    private String appName;

    /**
     * 近期任务的存储方案，默认为内存，适用于单实例的情况
     * 可配置为Redis/Memory/Mysql
     * 注意：配置文件中使用驼峰命名法会无法识别
     */
    private RepositoryStrategy recentJobRepository = RepositoryStrategy.MEMORY;

    /**
     * 一定时间以后才需要执行的任务，默认为内存，适用于单实例的情况
     * 可配置为Redis/Memory/Mysql
     * 注意：配置文件中使用驼峰命名法会无法识别
     */
    private RepositoryStrategy longTermJobRepository = RepositoryStrategy.MEMORY;

    /**
     * 执行线程池的配置
     */
    private ExecutorThreadPool executorPool = new ExecutorThreadPool();

    @Data
    public static class ExecutorThreadPool {
        /**
         * 核心线程数
         */
        private int coreSize = 5;

        /**
         * 最大线程数
         */
        private int maxSize = 50;

        /**
         * 待执行队列数
         */
        private int queueSize = 1000;
    }
}
