package com.bxm.component.oncejob.bootstrap;

import com.bxm.component.oncejob.job.JobPersistentObject;
import lombok.extern.slf4j.Slf4j;

import java.util.List;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;

/**
 * 每隔一秒执行一次，将分片好的任务提交到线程池进行执行操作
 *
 * @author liujia
 * @date 8/2/21 10:51 AM
 **/
@Slf4j
public class JobExecuteThread extends Thread {

    private JobExecutor jobExecutor;

    private AtomicBoolean threadToStop = new AtomicBoolean(false);

    public JobExecuteThread(JobExecutor jobExecutor) {
        super("once-job-executor");
        this.jobExecutor = jobExecutor;
    }

    @Override
    public void run() {
        while (!threadToStop.get()) {
            try {
                TimeUnit.MILLISECONDS.sleep(1000 - System.currentTimeMillis() % 1000);
            } catch (InterruptedException e) {
                if (!threadToStop.get()) {
                    log.error(e.getMessage(), e);
                }
            }

            try {
                List<JobPersistentObject> executeJobs = JobHolder.getCurrentSecondExecuteJobs();
                if (executeJobs == null) {
                    continue;
                }

                // 任务提交到线程池执行
                for (JobPersistentObject executeJob : executeJobs) {
                    jobExecutor.submit(executeJob);
                }
            } catch (Exception e) {
                log.error(e.getMessage(), e);
            }
        }
    }

    public void stopThread() {
        // 不再继续执行
        if (JobHolder.isEmpty()) {
            threadToStop.set(true);
        } else {
            log.info("存在预加载的调度任务，等待处理");
            while (true) {
                try {
                    TimeUnit.SECONDS.sleep(1);
                } catch (InterruptedException e) {
                    log.error(e.getMessage(), e);
                    threadToStop.set(true);
                }

                if (JobHolder.isEmpty()) {
                    threadToStop.set(true);
                    break;
                }
            }
            log.info("预加载任务处理完成");
        }

        log.info("停止处理本地预拉取的调度任务");

        // 停止处理本地任务
        jobExecutor.stop();

        try {
            // 等待1秒的执行时间
            TimeUnit.SECONDS.sleep(1L);
        } catch (InterruptedException e) {
            log.error(e.getMessage(), e);
        }

        if (Thread.currentThread().getState() != State.TERMINATED) {
            Thread.currentThread().interrupt();
        }

        log.info("关闭执行线程JobExecuteThread");
    }
}
