package com.bxm.component.oncejob.bootstrap;

import cn.hutool.core.date.DateUtil;
import cn.hutool.core.date.TimeInterval;
import com.bxm.component.oncejob.config.ComponentOnceJobConfigurationProperties;
import com.bxm.component.oncejob.counter.JobCounter;
import com.bxm.component.oncejob.enums.MissFireStrategy;
import com.bxm.component.oncejob.job.IOnceJobCallback;
import com.bxm.component.oncejob.job.JobPersistentObject;
import com.bxm.newidea.component.JSON;
import com.bxm.newidea.component.thread.NamedThreadFactory;
import com.bxm.newidea.component.tools.DateUtils;
import com.bxm.newidea.component.tools.ReflectionUtils;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.springframework.context.ApplicationContext;

import java.util.Date;
import java.util.Map;
import java.util.concurrent.LinkedBlockingDeque;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

/**
 * 任务执行器，将序列号任务进行处理，推送到线程池
 * 执行真正的业务逻辑处理，将接收的任务提交到线程池
 *
 * @author liujia
 * @date 8/2/21 10:55 AM
 **/
@Slf4j
class JobExecutor {

    private ComponentOnceJobConfigurationProperties properties;

    private ThreadPoolExecutor fastPoolExecutor;

    private Map<String, IOnceJobCallback> callbackMap;

    private ApplicationContext applicationContext;

    JobExecutor(ComponentOnceJobConfigurationProperties properties, ApplicationContext applicationContext) {
        this.properties = properties;
        this.applicationContext = applicationContext;
        callbackMap = Maps.newHashMap();
        init();
    }


    private void init() {
        ComponentOnceJobConfigurationProperties.ExecutorThreadPool threadPoolConfig = properties.getExecutorPool();
        fastPoolExecutor = new ThreadPoolExecutor(threadPoolConfig.getCoreSize(),
                threadPoolConfig.getMaxSize(),
                60L,
                TimeUnit.SECONDS,
                new LinkedBlockingDeque<>(threadPoolConfig.getQueueSize()),
                new NamedThreadFactory("once-job-executor"),
                (r, executor) -> log.error("任务数量超过了线程池的配置，请注意调整，当前配置：core:{},max:{},queue:{}",
                        threadPoolConfig.getCoreSize(),
                        threadPoolConfig.getMaxSize(),
                        threadPoolConfig.getQueueSize()));
    }

    @SuppressWarnings("unchecked")
    void submit(JobPersistentObject jobInfo) {
        JobCounter.addRunningCount();

        fastPoolExecutor.submit(() -> {
            IOnceJobCallback callback;
            try {
                callback = getCallback(jobInfo);
            } catch (ClassNotFoundException | IllegalAccessException | InstantiationException e) {
                JobCounter.addFailedCount();

                log.error(e.getMessage(), e);
                log.error("创建一次性任务回调类[{}]失败，请确定是否存在可用的空构造函数或注入到了spring context", jobInfo.getCallbackClassName());
                return;
            }

            try {

                long diffMills = jobInfo.getFireDate() - System.currentTimeMillis();

                if (log.isDebugEnabled()) {
                    log.debug("任务实际执行和设定时间相差：{}", diffMills);
                }

                if (Math.abs(diffMills) > 1000) {
                    JobCounter.addExpiredCount();
                    log.warn("任务[{}][{}]执行偏差时间超过了一秒钟,预计执行时间：[{}],实际执行时间：[{}]",
                            jobInfo.getJobId(),
                            callback.getClass().getName(),
                            DateUtils.formatDateTime(new Date(jobInfo.getFireDate())),
                            DateUtils.formatDateTime(new Date()));

                    if (MissFireStrategy.NONE.name().equals(jobInfo.getMissFireStrategy())) {
                        log.info("任务[{}]过期，不予执行", jobInfo.getJobId());
                        JobCounter.addFailedCount();
                        return;
                    }
                }

                TimeInterval timer = DateUtil.timer();

                boolean result = callback.execute(restoreParam(jobInfo, callback));

                if (timer.interval() > properties.getJobExpendWarningMills()) {
                    log.warn("任务回调[{}]执行超过了{}毫秒，请注意优化", jobInfo.getCallbackClassName(), properties.getJobExpendWarningMills());
                }

                if (result) {
                    if (log.isDebugEnabled()) {
                        log.debug("任务[{}]执行成功", jobInfo.getJobId());
                    }
                    JobCounter.addSuccessCount();
                } else {
                    JobCounter.addFailedCount();
                }
            } catch (Exception e) {
                log.error(e.getMessage(), e);
            }

            JobCounter.removeRunningCount();

            if (log.isDebugEnabled()) {
                log.debug("{}", JobCounter.state());
            }
        });
    }

    /**
     * 停止任务执行，在停止之前，处理所有已经提交的任务
     */
    void stop() {
        log.info("预备停止本地任务处理,待处理任务数量：{}", JobCounter.getRunningCount());

        try {
            // 尝试等待任务执行完成
            TimeUnit.MILLISECONDS.sleep(JobCounter.getRunningCount() * 100);
        } catch (InterruptedException e) {
            log.error(e.getMessage(), e);
        }

        fastPoolExecutor.shutdown();
        if (JobCounter.getRunningCount() > 0) {
            log.error("本地任务执行器关闭，剩余任务：{}", JobCounter.getRunningCount());
        } else {
            log.info("本地任务执行器关闭");
        }
    }

    private Object restoreParam(JobPersistentObject jobInfo, IOnceJobCallback callback) {
        Class<?> firstGenericType = ReflectionUtils.getFirstGenericType(callback.getClass());
        String jsonString = jobInfo.getParamJson();
        return JSON.parseObject(jsonString, firstGenericType);
    }

    private IOnceJobCallback getCallback(JobPersistentObject jobInfo) throws ClassNotFoundException, IllegalAccessException, InstantiationException {
        String callbackClassName = jobInfo.getCallbackClassName();
        IOnceJobCallback callback = callbackMap.get(callbackClassName);
        if (callback == null) {
            Class<?> callbackClass = Class.forName(callbackClassName);

            Map<String, ?> beansOfType = applicationContext.getBeansOfType(callbackClass);
            if (beansOfType.size() != 0) {
                Object callbackSpringBean = beansOfType.values().iterator().next();
                if (callbackSpringBean instanceof IOnceJobCallback) {
                    callback = (IOnceJobCallback) callbackSpringBean;
                }
            }

            if (null == callback) {
                callback = (IOnceJobCallback) callbackClass.newInstance();
            }

            callbackMap.put(callbackClassName, callback);
        }
        return callback;
    }
}
