package com.bxm.component.oncejob.bootstrap;

import com.bxm.component.oncejob.counter.JobCounter;
import com.bxm.component.oncejob.enums.MissFireStrategy;
import com.bxm.component.oncejob.job.JobPersistentObject;
import com.bxm.newidea.component.tools.StringUtils;
import com.google.common.collect.Maps;
import lombok.experimental.UtilityClass;
import lombok.extern.slf4j.Slf4j;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * 内存中存储任务信息
 * 将预加载的任务存储到本地，通过{@link JobExecuteThread}定时调度执行任务
 *
 * @author liujia
 * @date 7/30/21 5:36 PM
 **/
@UtilityClass
@Slf4j
public class JobHolder {

    /**
     * 保存每一秒需要执行的任务
     * key：当前秒数
     * value：当前这一秒需要执行的任务数
     */
    private static Map<Long, List<JobPersistentObject>> jobQueueMap = Maps.newConcurrentMap();

    /**
     * 将任务推送到本地执行队列，根据执行时间，放到对应的执行秒数中
     *
     * @param jobInfo 任务信息
     */
    public static void pushJob(JobPersistentObject jobInfo) {
        if (log.isDebugEnabled()) {
            log.debug("添加任务到本地队列：{}", jobInfo.getJobId());
        }

        long executeSecond = jobInfo.getFireDate() / 1000;
        List<JobPersistentObject> waitExecuteJobs = jobQueueMap.getOrDefault(executeSecond, new ArrayList<>());
        waitExecuteJobs.add(jobInfo);

        JobCounter.addQueueCount();

        jobQueueMap.put(executeSecond, waitExecuteJobs);
    }

    public boolean isEmpty() {
        return jobQueueMap.size() == 0;
    }

    /**
     * 删除本地队列中的特定任务
     *
     * @param jobId 任务ID
     * @return 是否删除成功
     */
    public static boolean removeJob(String jobId) {
        for (List<JobPersistentObject> jobs : jobQueueMap.values()) {
            for (JobPersistentObject job : jobs) {
                if (StringUtils.equals(jobId, job.getJobId())) {
                    jobs.remove(job);

                    JobCounter.consumeQueueCount(1);
                    return true;
                }
            }
        }

        return false;
    }

    /**
     * 获取并移除当前这一秒需要执行的任务
     *
     * @return 执行任务列表
     */
    public static List<JobPersistentObject> getCurrentSecondExecuteJobs() {
        long executeSecond = System.currentTimeMillis() / 1000;
        List<JobPersistentObject> result = jobQueueMap.remove(executeSecond);

        if (log.isDebugEnabled()) {
            if (null != result && result.size() > 0) {
                log.debug("[{}]秒需要执行的任务有[{}]个", executeSecond, result.size());
            } else {
                log.debug("[{}]秒没有要执行的任务", executeSecond);
            }
        }

        if (null != result) {
            JobCounter.consumeQueueCount(result.size());
        }

        return result;
    }

    /**
     * 删除队列中的脏数据，如果出现了日志，说明逻辑有问题
     * 检查是否存在比三秒之前小的数据
     */
    public static void clearDirty() {
        long currentSecond = System.currentTimeMillis() / 1000;
        long nextSecond = currentSecond + 1;
        long preSecond = currentSecond - 1;

        for (Map.Entry<Long, List<JobPersistentObject>> entry : jobQueueMap.entrySet()) {
            if (entry.getKey() < preSecond) {
                log.error("[{}]秒的任务未能正确执行，共：[{}]个，现在进行补充执行", entry.getKey(), entry.getValue().size());
                List<JobPersistentObject> jobs = jobQueueMap.remove(entry.getKey());

                jobs = jobs.stream().filter(job -> MissFireStrategy.REMAIN.name().equals(job.getMissFireStrategy()))
                        .collect(Collectors.toList());

                // 将错过执行时间的任务放到下一秒继续执行
                List<JobPersistentObject> waitExecuteJobs = jobQueueMap.getOrDefault(nextSecond, new ArrayList<>());
                waitExecuteJobs.addAll(jobs);
                jobQueueMap.put(nextSecond, waitExecuteJobs);
            }
        }
    }
}
