package com.bxm.component.oncejob.bootstrap;

import com.bxm.component.oncejob.config.ComponentOnceJobConfigurationProperties;
import com.bxm.component.oncejob.storage.JobRepository;
import com.bxm.component.oncejob.storage.LongTermJobRepository;
import com.bxm.component.oncejob.storage.RecentJobRepository;
import com.bxm.newidea.component.thread.NamedThreadFactory;
import com.bxm.newidea.component.thread.UncaughtExceptionHandler;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.boot.ApplicationArguments;
import org.springframework.boot.ApplicationRunner;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;

import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

/**
 * 定时任务的定时处理启动器
 *
 * @author liujia
 * @date 7/30/21 10:42 AM
 **/
@Slf4j
public class OnceJobTimerBootstrapper implements ApplicationRunner, DisposableBean, ApplicationContextAware {

    private ComponentOnceJobConfigurationProperties properties;

    private RecentJobRepository recentJobRepository;

    private LongTermJobRepository longTermJobRepository;

    private JobRepository jobRepository;

    private ScheduledExecutorService fetchJobExecutor;

    private ScheduledExecutorService clearDirtyExecutor;

    private ScheduledExecutorService fetchLongTermExecutor;

    private JobExecuteThread jobExecuteThread;

    private ApplicationContext applicationContext;

    public OnceJobTimerBootstrapper(ComponentOnceJobConfigurationProperties properties,
                                    RecentJobRepository recentJobRepository,
                                    LongTermJobRepository longTermJobRepository,
                                    JobRepository jobRepository) {
        this.properties = properties;
        this.recentJobRepository = recentJobRepository;
        this.longTermJobRepository = longTermJobRepository;
        this.jobRepository = jobRepository;
    }

    @Override
    public void run(ApplicationArguments args) {
        if (!properties.isEnable()) {
            log.info("关闭了一次性定时任务处理，不进行相关任务的初始化");
            return;
        }

        initFetchJobExecutor();
        initFetchLongTermExecutor();
        initClearDirtyExecutor();
        initJobExecutor();
    }

    /**
     * 初始化脏数据清理线程，定期清理各个环境的脏数据
     */
    private void initClearDirtyExecutor() {
        clearDirtyExecutor = new ScheduledThreadPoolExecutor(1,
                new NamedThreadFactory("once-job-clear-dirty", true));

        clearDirtyExecutor.scheduleAtFixedRate(() -> jobRepository.clearDirty(),
                1000L,
                properties.getClearDirtyIntervalMills(),
                TimeUnit.MILLISECONDS);
    }

    /**
     * 从即将执行的任务库中拉取到本地执行队列
     * 减少远程交互，提高本地运行效率
     */
    private void initFetchJobExecutor() {
        fetchJobExecutor = new ScheduledThreadPoolExecutor(1,
                new NamedThreadFactory("once-job-fetch",
                        true));

        long delay = System.currentTimeMillis() % properties.getFetchIntervalMills();
        FetchRecentJobRunner fetchRecentJobRunner = new FetchRecentJobRunner(recentJobRepository, properties);
        fetchJobExecutor.scheduleAtFixedRate(fetchRecentJobRunner,
                delay,
                properties.getFetchIntervalMills(),
                TimeUnit.MILLISECONDS);
    }

    /**
     * 从长期任务库中拉取到即将执行的任务库
     * 减少即将执行的任务库的存储数量，针对不同时态的任务做不同的归档处理，提高效率
     */
    private void initFetchLongTermExecutor() {
        fetchLongTermExecutor = new ScheduledThreadPoolExecutor(1,
                new NamedThreadFactory("once-job-fetch-long",
                        true));
        FetchLongTermJobRunner runner = new FetchLongTermJobRunner(recentJobRepository, longTermJobRepository, properties);

        long delay = System.currentTimeMillis() % properties.getFetchLongTermJobMills();

        fetchLongTermExecutor.scheduleAtFixedRate(runner,
                delay,
                properties.getFetchLongTermJobMills(),
                TimeUnit.MILLISECONDS);
    }

    /**
     * 初始化任务真正的执行器，每一秒执行一次，将这一秒对应的任务提交到线程池进行异步执行
     */
    private void initJobExecutor() {
        // 定时执行线程
        JobExecutor jobExecutor = new JobExecutor(properties, applicationContext);

        jobExecuteThread = new JobExecuteThread(jobExecutor);
        jobExecuteThread.setDaemon(true);
        jobExecuteThread.setUncaughtExceptionHandler(new UncaughtExceptionHandler());
        jobExecuteThread.start();
    }

    @Override
    public void destroy() {
        fetchJobExecutor.shutdown();
        log.info("fetchJobExecutor关闭完成");

        clearDirtyExecutor.shutdown();
        log.info("clearDirtyExecutor关闭完成");

        fetchLongTermExecutor.shutdown();
        log.info("fetchLongTermExecutor关闭完成");

        // 正在处理的任务，如果强制关闭线程则可能会丢失正在执行的任务
        jobExecuteThread.stopThread();
    }

    @Override
    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
        this.applicationContext = applicationContext;
    }
}
