package com.bxm.component.oncejob.config;

import com.bxm.component.oncejob.bootstrap.OnceJobTimerBootstrapper;
import com.bxm.component.oncejob.storage.JobRepository;
import com.bxm.component.oncejob.storage.LongTermJobRepository;
import com.bxm.component.oncejob.storage.RecentJobRepository;
import com.bxm.component.oncejob.storage.impl.JobRepositoryImpl;
import com.bxm.component.oncejob.storage.memory.MemoryLongTermJobRepositoryImpl;
import com.bxm.component.oncejob.storage.memory.MemoryRecentJobRepositoryImpl;
import com.bxm.component.oncejob.storage.redis.RedisLongTermJobRepositoryImpl;
import com.bxm.component.oncejob.storage.redis.RedisRecentJobRepositoryImpl;
import com.bxm.newidea.component.redis.DistributedLock;
import com.bxm.newidea.component.redis.RedisHashMapAdapter;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.newidea.component.redis.RedisZSetAdapter;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.autoconfigure.AutoConfigureOrder;
import org.springframework.boot.autoconfigure.condition.ConditionalOnBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Primary;
import org.springframework.core.Ordered;

/**
 * 一次性调度任务自动装配对象
 *
 * @author liujia
 * @date 7/29/21 4:58 PM
 **/
@Configuration(proxyBeanMethods = false)
@EnableConfigurationProperties(ComponentOnceJobConfigurationProperties.class)
@ConditionalOnProperty(prefix = "component.once-job", name = "enable", matchIfMissing = true)
@AllArgsConstructor
@Slf4j
@AutoConfigureOrder(Ordered.LOWEST_PRECEDENCE)
public class ComponentOnceJobAutoConfiguration {

    private ComponentOnceJobConfigurationProperties properties;

    @Bean
    public OnceJobTimerBootstrapper bootstrapper(JobRepository jobRepository,
                                                 RecentJobRepository recentJobRepository,
                                                 LongTermJobRepository longTermJobRepository) {
        return new OnceJobTimerBootstrapper(properties, recentJobRepository, longTermJobRepository, jobRepository);
    }

    /**
     * 综合任务存储接口，根据任务进行不同的分发处理
     *
     * @param recentJobRepository   近期任务存储
     * @param longTermJobRepository 长期任务存储
     * @return 任务存储接口
     */
    @Primary
    @Bean
    public JobRepository jobRepository(RecentJobRepository recentJobRepository, LongTermJobRepository longTermJobRepository) {
        return new JobRepositoryImpl(recentJobRepository, longTermJobRepository, properties);
    }

    /**
     * 基于redis实现的近期调度任务存储方案
     *
     * @param redisHashMapAdapter redisHashMapAdapter
     * @param distributedLock     distributedLock
     * @return redis的近期任务存储
     */
    @Bean
    @ConditionalOnProperty(prefix = "component.once-job", name = "recentJobRepository", havingValue = "redis")
    @ConditionalOnBean({RedisHashMapAdapter.class, RedisStringAdapter.class})
    public RecentJobRepository redisRecentJobRepository(RedisHashMapAdapter redisHashMapAdapter,
                                                        DistributedLock distributedLock,
                                                        RedisZSetAdapter redisZSetAdapter) {
        log.info("启用redis作为一次性任务的短期存储方案");
        return new RedisRecentJobRepositoryImpl(properties, redisHashMapAdapter, distributedLock, redisZSetAdapter);
    }

    @Bean
    @ConditionalOnProperty(prefix = "component.once-job", name = "longTermJobRepository", havingValue = "redis")
    @ConditionalOnBean({RedisHashMapAdapter.class, RedisZSetAdapter.class})
    public LongTermJobRepository redisLongTermJobRepository(RedisHashMapAdapter redisHashMapAdapter,
                                                            RedisZSetAdapter redisZSetAdapter,
                                                            DistributedLock distributedLock) {
        log.info("启用redis作为一次性任务的长期存储方案");
        return new RedisLongTermJobRepositoryImpl(redisHashMapAdapter, properties, distributedLock, redisZSetAdapter);
    }

    @Bean
    @ConditionalOnProperty(prefix = "component.once-job", name = "recentJobRepository", havingValue = "memory", matchIfMissing = true)
    public RecentJobRepository memoryRecentJobRepository() {
        log.warn("启用memory作为一次性任务的短期存储方案，实例停止时任务将会丢失");
        return new MemoryRecentJobRepositoryImpl(properties);
    }

    @Bean
    @ConditionalOnProperty(prefix = "component.once-job", name = "longTermJobRepository", havingValue = "memory", matchIfMissing = true)
    public LongTermJobRepository memoryLongTermJobRepository(RecentJobRepository recentJobRepository) {
        log.warn("启用memory作为一次性任务的长期存储方案，实例停止时任务将会丢失");
        return new MemoryLongTermJobRepositoryImpl(recentJobRepository);
    }
}
