package com.bxm.component.oncejob.storage.impl;

import com.bxm.component.oncejob.bootstrap.JobHolder;
import com.bxm.component.oncejob.config.ComponentOnceJobConfigurationProperties;
import com.bxm.component.oncejob.converter.JobConverter;
import com.bxm.component.oncejob.counter.JobCounter;
import com.bxm.component.oncejob.job.JobPersistentObject;
import com.bxm.component.oncejob.job.OnceJobDefinition;
import com.bxm.component.oncejob.storage.JobRepository;
import com.bxm.component.oncejob.storage.LongTermJobRepository;
import com.bxm.component.oncejob.storage.RecentJobRepository;
import com.bxm.newidea.component.JSON;
import com.google.common.collect.Lists;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;

import java.util.List;

/**
 * 对外的代理实现，对短期和长期的任务进行管理
 *
 * @author liujia
 * @date 7/30/21 10:25 AM
 **/
@AllArgsConstructor
@Slf4j
public class JobRepositoryImpl implements JobRepository {

    private RecentJobRepository recentJobRepository;

    private LongTermJobRepository longTermJobRepository;

    private ComponentOnceJobConfigurationProperties properties;

    @Override
    public String saveJob(OnceJobDefinition definition) {
        if (log.isDebugEnabled()) {
            log.debug("创建一次性定时任务：{}", JSON.toJSONString(definition));
        }

        JobCounter.addCreateCount();

        // 触发事件小于拉取间隔的，表示马上执行，直接丢入到队列中
        if (definition.getFireDate().getTime() - System.currentTimeMillis() <= properties.getFetchIntervalMills()) {
            JobPersistentObject persistentObject = JobConverter.convert(definition);
            JobHolder.pushJob(persistentObject);
            return persistentObject.getJobId();
        }

        if (definition.getFireDate().getTime() - System.currentTimeMillis() > properties.getFetchLongTermJobMills()) {
            // 进行长期存储
            return longTermJobRepository.saveJob(definition);
        }

        return recentJobRepository.saveJob(definition);
    }

    @Override
    public boolean removeJob(String jobId) {
        // 移除本地队列中的任务
        boolean removed = JobHolder.removeJob(jobId);

        if (!removed) {
            // 移除近期即将执行的任务
            removed = recentJobRepository.removeJob(jobId);

            if (!removed) {
                // 移除长期存储的任务
                removed = longTermJobRepository.removeJob(jobId);
            }
        }

        return removed;
    }

    @Override
    public List<JobPersistentObject> clearDirty() {
        if (log.isDebugEnabled()) {
            log.debug("执行脏数据清理");
        }

        List<JobPersistentObject> jobPersistentObjects = recentJobRepository.clearDirty();
        jobPersistentObjects.addAll(longTermJobRepository.clearDirty());

        List<JobPersistentObject> recentList = Lists.newArrayList();

        for (JobPersistentObject jobPersistentObject : jobPersistentObjects) {
            if (jobPersistentObject.getFireDate() - System.currentTimeMillis() <= properties.getFetchIntervalMills()) {
                JobHolder.pushJob(jobPersistentObject);
            } else {
                recentList.add(jobPersistentObject);
            }
        }

        recentJobRepository.pushAll(recentList);

        JobHolder.clearDirty();

        log.info("{}", JobCounter.state());

        return jobPersistentObjects;
    }
}
