package com.bxm.component.oncejob.storage.redis;

import com.bxm.component.oncejob.config.ComponentOnceJobConfigurationProperties;
import com.bxm.component.oncejob.converter.JobConverter;
import com.bxm.component.oncejob.job.JobPersistentObject;
import com.bxm.component.oncejob.job.OnceJobDefinition;
import com.bxm.component.oncejob.storage.RecentJobRepository;
import com.bxm.component.oncejob.utils.FragmentIndexUtil;
import com.bxm.component.oncejob.utils.JobIdUtil;
import com.bxm.newidea.component.entity.TypedTuple;
import com.bxm.newidea.component.redis.DistributedLock;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisHashMapAdapter;
import com.bxm.newidea.component.redis.RedisZSetAdapter;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;

import java.util.List;
import java.util.Set;

import static com.bxm.component.oncejob.constant.OnceJobRedisKey.*;

/**
 * 将即将执行的任务存储到redis中
 *
 * @author liujia
 * @date 7/30/21 10:23 AM
 **/
@Slf4j
public class RedisRecentJobRepositoryImpl implements RecentJobRepository {

    private ComponentOnceJobConfigurationProperties properties;

    private RedisHashMapAdapter redisHashMapAdapter;

    private DistributedLock distributedLock;

    private RedisZSetAdapter redisZSetAdapter;

    public RedisRecentJobRepositoryImpl(ComponentOnceJobConfigurationProperties properties,
                                        RedisHashMapAdapter redisHashMapAdapter,
                                        DistributedLock distributedLock,
                                        RedisZSetAdapter redisZSetAdapter) {
        this.properties = properties;
        this.redisHashMapAdapter = redisHashMapAdapter;
        this.distributedLock = distributedLock;
        this.redisZSetAdapter = redisZSetAdapter;
    }

    @Override
    public String saveJob(OnceJobDefinition definition) {
        // 进行短期存储
        return savePersistentObject(JobConverter.convert(definition));
    }

    private String savePersistentObject(JobPersistentObject persistentObject) {
        String jobId = persistentObject.getJobId();

        // 任务分片，用于定期拉取
        long fragmentIndex = FragmentIndexUtil.getFragmentIndex(persistentObject.getFireDate(),
                properties.getFetchIntervalMills());
        KeyGenerator key = getJobFragmentIndexKey(fragmentIndex);
        redisHashMapAdapter.put(key, jobId, persistentObject);

        redisZSetAdapter.add(JOB_FRAGMENT_INDEX_KEY, fragmentIndex, fragmentIndex);

        return jobId;
    }

    @Override
    public void pushAll(List<JobPersistentObject> jobs) {
        for (JobPersistentObject job : jobs) {
            savePersistentObject(job);
        }
    }

    /**
     * 创建应用维度的任务分片
     *
     * @param fragmentIndex 任务分片索引
     * @return 任务分片key
     */
    private KeyGenerator getJobFragmentIndexKey(long fragmentIndex) {
        return JOB_FRAGMENT_KEY.copy().appendKey(properties.getAppName()).appendKey(fragmentIndex);
    }


    @Override
    public boolean removeJob(String jobId) {
        if (log.isDebugEnabled()) {
            log.debug("移除一次性任务:{}", jobId);
        }

        long fragmentIndex = FragmentIndexUtil.getFragmentIndex(JobIdUtil.getFireMills(jobId),
                properties.getFetchIntervalMills());

        // 删除任务信息
        KeyGenerator key = getJobFragmentIndexKey(fragmentIndex);
        return redisHashMapAdapter.remove(key, jobId) > 0;
    }

    @Override
    public List<JobPersistentObject> load(long lastFireTimeMills, int num) {
        long fragmentIndex = FragmentIndexUtil.getFragmentIndex(lastFireTimeMills, properties.getFetchIntervalMills());

        String resource = JOB_FRAGMENT_INDEX_LOCK_KEY.copy()
                .appendKey(properties.getAppName())
                .appendKey(fragmentIndex).gen();

        List<JobPersistentObject> preFetch = Lists.newArrayList();

        if (distributedLock.lock(resource)) {
            preFetch = loadInterval(fragmentIndex);

            if (log.isDebugEnabled()) {
                log.debug("预加载一次性任务，加载时间点:[{}],加载片区:[{}],加载数量:[{}]",
                        lastFireTimeMills, fragmentIndex, preFetch.size());
            }

            distributedLock.unlock(resource);
        }

        return preFetch;
    }

    private List<JobPersistentObject> loadInterval(long fragmentIndex) {
        KeyGenerator key = getJobFragmentIndexKey(fragmentIndex);

        List<JobPersistentObject> values = redisHashMapAdapter.values(key, JobPersistentObject.class);
        // 可以考虑增加一个严格模式，区分处理，在执行成功后才处理
        redisHashMapAdapter.remove(key);
        redisZSetAdapter.remove(JOB_FRAGMENT_INDEX_KEY, fragmentIndex);

        return values;
    }

    @Override
    public List<JobPersistentObject> clearDirty() {
        String lockKey = JOB_RECENT_DIRTY_LOCK_KEY.gen();
        List<JobPersistentObject> result = Lists.newArrayList();

        if (distributedLock.lock(lockKey)) {
            long fragmentIndex = FragmentIndexUtil.getFragmentIndex(System.currentTimeMillis(), properties.getFetchIntervalMills());

            Set<TypedTuple<Long>> typedTuples = redisZSetAdapter.rangeByScoreWithScores(JOB_FRAGMENT_INDEX_KEY,
                    0,
                    fragmentIndex,
                    true,
                    Long.class);

            log.info("发现[{}]个未处理的时间片段", typedTuples.size());

            for (TypedTuple<Long> typedTuple : typedTuples) {
                result.addAll(loadInterval(typedTuple.getValue()));
            }

            distributedLock.unlock(lockKey);
        } else {
            log.warn("其他实例正在处理脏数据，放弃本次处理");
        }

        return result;
    }
}
