package com.bxm.newidea.component.oss.service.impl;

import java.io.*;
import java.util.Date;

import com.aliyun.oss.OSSClient;
import com.aliyun.oss.model.GetObjectRequest;
import com.aliyun.oss.model.OSSObject;
import com.aliyun.oss.model.ObjectMetadata;
import com.aliyun.oss.model.PutObjectResult;
import com.bxm.newidea.component.enums.FileTypeEnum;
import com.bxm.newidea.component.oss.config.AliyunOssProperties;
import com.bxm.newidea.component.oss.service.AliyunOSSService;
import com.bxm.newidea.component.tools.DateUtils;
import com.bxm.newidea.component.tools.FileUtils;
import com.bxm.newidea.component.tools.StringUtils;
import com.bxm.newidea.component.tools.UUIDUtils;
import org.apache.commons.io.IOUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.web.multipart.MultipartFile;

import java.io.*;
import java.util.Date;

@Component("aliyunOssService")
public class AliyunOSSServiceImpl implements AliyunOSSService {

    private final static Logger log = LoggerFactory.getLogger(AliyunOSSService.class);

    private final static String separator = "/";

    private AliyunOssProperties aliyunOssProperties;

    @Autowired
    public AliyunOSSServiceImpl(AliyunOssProperties aliyunOssProperties) {
        this.aliyunOssProperties = aliyunOssProperties;
    }

    @Override
    public String getOSSFileRoot() {
        if (StringUtils.isNotBlank(this.aliyunOssProperties.getCdnUrl())) {
            return this.aliyunOssProperties.getCdnUrl() + "/" + this.aliyunOssProperties.getFolder() + "/";
        }
        return "https://" + this.aliyunOssProperties.getBucket() + "."
                + this.aliyunOssProperties.getEndpoint() + "/"
                + this.aliyunOssProperties.getFolder() + "/";
    }

    @Override
    public String[] upload(MultipartFile... multipartFiles) {
        String[] strings = new String[multipartFiles.length];
        try {
            int i = 0;
            for (MultipartFile multipartFile : multipartFiles) {
                String fileName = multipartFile.getOriginalFilename();
                long contentLength = multipartFile.getSize();
                //获取文件的contentType
                String contentType = multipartFile.getContentType();

                if (StringUtils.isBlank(contentType)) {
                    contentType = "text/plain";
                }

                String ossPath = generateOssPathByFile(fileName);

                PutObjectResult result = upload(multipartFile.getInputStream(), contentLength, contentType, ossPath);

                strings[i++] = getOSSFileRoot() + ossPath;
            }
        } catch (IOException e) {
            log.error("upload multipartFiles error," + e.getMessage(), e);
        }
        return strings;
    }

    @Override
    public String upload(MultipartFile multipartFile) {
        return upload(new MultipartFile[]{multipartFile})[0];
    }

    @Override
    public String upload(MultipartFile multipartFile, String path) {
        try {
            long contentLength = multipartFile.getSize();

            //获取文件的contentType
            String contentType = multipartFile.getContentType();

            if (StringUtils.isBlank(contentType)) {
                contentType = "text/plain";
            }

            String ossPath = generateOssPath(path, multipartFile.getOriginalFilename());

            PutObjectResult result = upload(multipartFile.getInputStream(), contentLength, contentType, ossPath);

            return getOSSFileRoot() + ossPath;
        } catch (Exception e) {
            log.error("upload file error:" + e.getMessage(), e);
        }
        return null;
    }

    @Override
    public String[] upload(File... files) {
        String[] uploadUrlArray = new String[files.length];
        try {
            int i = 0;
            for (File file : files) {
                //获取file.length
                long contentLength = file.length();

                //获取文件的contentType
                String suffix = FileUtils.getFileExtension(file);

                String contentType = FileTypeEnum.getContentType(suffix);
                if (StringUtils.isBlank(contentType)) {
                    contentType = "text/plain";
                }

                String fileName = file.getName();

                //生成上传的地址
                String ossPath = generateOssPathByFile(fileName);

                PutObjectResult result = upload(new FileInputStream(file), contentLength, contentType, ossPath);

                uploadUrlArray[i++] = getOSSFileRoot() + ossPath;
            }
        } catch (IOException e) {
            log.error("upload files error", e);
        }
        return uploadUrlArray;
    }

    @Override
    public String upload(File file, String path) {
        try {
            //获取file.length
            long contentLength = file.length();

            //获取文件的contentType
            String suffix = FileUtils.getFileExtension(file);
            String contentType = FileTypeEnum.getContentType(suffix);

            if (StringUtils.isBlank(contentType)) {
                contentType = "text/plain";
            }

            //生成上传的地址
            String ossPath = generateOssPath(path, file.getName());

            PutObjectResult result = upload(new FileInputStream(file), contentLength, contentType, ossPath);

            return getOSSFileRoot() + ossPath;
        } catch (Exception e) {
            log.error("upload file error:" + e.getMessage(), e);
        }
        return null;
    }

    @Override
    public String upload(InputStream stream, String path) {
        OSSClient client = getOssClient();

        String ossFilePath = generateOssPathByPath(path);
        String uploadPath = this.aliyunOssProperties.getFolder() + separator + ossFilePath;

        PutObjectResult result = client.putObject(this.aliyunOssProperties.getBucket(), uploadPath, stream);
        client.shutdown();
        return getOSSFileRoot() + ossFilePath;
    }

    @Override
    public String upload(byte[] data, String path) {
        try {
            long contentLength = data.length;
            //生成路径
            String contentType = FileTypeEnum.getContentType(StringUtils.substringAfterLast(path, "."));
            if (StringUtils.isBlank(contentType)) {
                contentType = "text/plain";
            }
            String ossPath = generateOssPathByPath(path);
            PutObjectResult result = upload(new ByteArrayInputStream(data), contentLength, contentType, ossPath);
            return getOSSFileRoot() + ossPath;
        } catch (Exception e) {
            log.error("字节流上传文件发生错误：" + e.getMessage(), e);
        }
        return null;
    }

    @Override
    public Boolean download(String path, String filePath) {
        try {
            OSSClient ossClient = getOssClient();
            if (path.startsWith(separator)) {
                path = path.substring(1);
            }
            path = this.aliyunOssProperties.getFolder() + separator + path;
            GetObjectRequest getObjectRequest = new GetObjectRequest(this.aliyunOssProperties.getBucket(), path);
            ossClient.getObject(getObjectRequest, new File(filePath));
            ossClient.shutdown();
            return true;
        } catch (Exception e) {
            log.error("download error", e);
            return false;
        }

    }

    /**
     * 根据上传的路径生成oss路径
     */
    private String generateOssPathByPath(String path) {
        String fileName = generateFileNameByPath(path);
        return generateOssPath(path, fileName);
    }

    /**
     * 根据文件名称生成oss路径
     */
    private String generateOssPathByFile(String fileName) {
        if (StringUtils.isNotBlank(fileName)) {
            fileName = UUIDUtils.nextID() + "." + generateSuffix(fileName);
        }
        if (StringUtils.isBlank(fileName)) {
            fileName = UUIDUtils.nextID();
        }
        return generateOssPath(null, fileName);
    }

    /**
     * 生成oss路径
     */
    private String generateOssPath(String path, String fileName) {
        path = handlePath(path);
        if (StringUtils.isBlank(path)) {
            path = generateDirByFileName(fileName).append(fileName).toString();
            return path;
        }

        //path为文件名
        if (!path.contains(separator)) {
            path = generateDirByFileName(fileName).append(path).toString();
            return path;
        }

        //path为目录
        if (path.endsWith(separator)) {
            String suffix = generateSuffix(fileName);
            path = path + UUIDUtils.nextID() + "." + suffix;
            return path;
        }

        return path;
    }

    /**
     * 处理path
     */
    private String handlePath(String path) {
        if (StringUtils.isBlank(path)) {
            return path;
        }

        path = path.replaceAll("\\\\", separator);
        if (path.startsWith(separator)) {
            path = path.substring(1);
        }

        return path;
    }

    /**
     * 根据path生成文件名(主要用于不能获取到文件名称)
     */
    private String generateFileNameByPath(String path) {
        path = handlePath(path);

        //path为空随机生成文件名
        if (StringUtils.isBlank(path)) {
            return UUIDUtils.nextID();
        }

        //path为文件名直接返回path
        if (!path.contains(separator)) {
            return path;
        }

        //path为目录随机生成文件名
        if (path.endsWith(separator)) {
            return UUIDUtils.nextID();
        }

        //path包含路径和文件名返回path的文件名
        return StringUtils.substringAfterLast(path, separator);
    }

    /**
     * 根据文件名设置文件上传的目录
     */
    private StringBuilder generateDirByFileName(String fileName) {
        String suffix = generateSuffix(fileName);
        String todayStr = DateUtils.PATTERN_NO_DELIMITER_FORMAT.get().format(new Date());
        return new StringBuilder().append(suffix).append(separator)
                .append(todayStr).append(separator);
    }

    /**
     * 获取文件前缀
     * @param fileName
     * @return
     */
    private String generateSuffix(String fileName) {
        String suffix = FileUtils.getFilextension(fileName);
        String contentType = FileTypeEnum.getContentType(suffix);
        if (StringUtils.isBlank(contentType)) {
            suffix = "txt";
        }

        return suffix;
    }

    /**
     * 获取oss实例
     * @return
     */
    private OSSClient getOssClient() {
        return new OSSClient(this.aliyunOssProperties.getEndpoint(),
                this.aliyunOssProperties.getAccessKey(),
                this.aliyunOssProperties.getAccessSecret());
    }

    /**
     * 上传文件到OSS.</br>
     * @param inputStream   输入流
     * @param contentLength 文件大小
     * @param contentType   文件类型
     * @param path          保存的资源路径全称
     */
    private PutObjectResult upload(InputStream inputStream, long contentLength, String contentType, String path) {
        try {
            // 创建上传Object的Metadata
            ObjectMetadata meta = new ObjectMetadata();
            // 必须设置ContentLength
            meta.setContentLength(contentLength);
            meta.setContentType(contentType);

            OSSClient ossClient = getOssClient();
            // 上传Object.
            path = this.aliyunOssProperties.getFolder() + separator + path;

            PutObjectResult result = ossClient.putObject(this.aliyunOssProperties.getBucket(), path, inputStream, meta);

            ossClient.shutdown();
            return result;
        } catch (Exception e) {
            throw e;
        } finally {
            IOUtils.closeQuietly(inputStream);
        }
    }

    @Override
    public String getOssContentByPath(String path) {
        OSSClient client = getOssClient();
        path = this.aliyunOssProperties.getFolder() + separator + path;
        StringBuilder stringBuilder = new StringBuilder();
        try(
                OSSObject ossObject = client.getObject(aliyunOssProperties.getBucket(),path);
                InputStream in = ossObject.getObjectContent();
                InputStreamReader inputStreamReader = new InputStreamReader(in);
                BufferedReader reader = new BufferedReader(inputStreamReader)
        ){
            String str;
            while ((str = reader.readLine())!= null  ) {
                stringBuilder.append(str);
            }
        }catch (Exception e){
            log.error("oss获取文件异常",e);
        }finally {
            // 关闭OSSClient。
            client.shutdown();
        }

        return stringBuilder.toString();

    }
}
