package com.bxm.newidea.component.payment.service;

import com.bxm.newidea.component.payment.request.PaymentRequest;
import com.bxm.newidea.component.payment.response.PaymentResponse;
import com.google.common.base.Joiner;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.List;
import java.util.Map;

/**
 * 支付动作执行分发类
 *
 * @author liujia
 */
@Slf4j
@Component
public class PaymentActionDispatcher {

    /**
     * 执行器映射MAP
     * key：关键参数的字符串组合
     * value：执行器的实现类
     */
    private Map<String, IPaymentAction> actionMap;

    @Autowired
    public PaymentActionDispatcher(List<IPaymentAction> actionList) {
        actionMap = Maps.newHashMap();

        for (IPaymentAction paymentAction : actionList) {
            String key = buildKey(paymentAction);

            if (actionMap.containsKey(key)) {
                log.error("{}已经存在,存在重复定义", key);
            }
            actionMap.put(key, paymentAction);
        }
    }

    /**
     * 执行具体的支付动作
     * 根据请求参数找到具体的执行类，进行对应的逻辑处理
     *
     * @return 如果参数异常将返回null
     */
    public <T extends PaymentResponse> T exec(PaymentRequest<T> request) {

        if (request == null || !request.verify()) {
            log.error("支付请求参数设置错误，请多看代码注释！！！！");
            return null;
        }

        String key = buildKey(request);

        IPaymentAction action = actionMap.get(key);
        if (null == action) {
            return null;
        }

        return (T) action.exec(request);
    }

    private String buildKey(PaymentRequest request) {
        return Joiner.on(":").join(request.getPlatform().name(),
                request.getMode().name(),
                request.getAction().name());
    }

    private String buildKey(IPaymentAction paymentAction) {
        return Joiner.on(":").join(paymentAction.matchPlatform().name(),
                paymentAction.matchMode().name(),
                paymentAction.matchAction().name());
    }
}
