package com.bxm.newidea.component.payment.config;

import com.alipay.api.AlipayClient;
import com.alipay.api.DefaultAlipayClient;
import com.bxm.component.httpclient.utils.OkHttpUtils;
import com.github.binarywang.wxpay.config.WxPayConfig;
import com.github.binarywang.wxpay.exception.WxPayException;
import com.github.binarywang.wxpay.service.WxPayService;
import com.github.binarywang.wxpay.service.impl.WxPayServiceImpl;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;

import java.io.File;
import java.util.Map;

import static org.apache.commons.lang3.StringUtils.startsWith;

/**
 * 支付配置构建工厂类
 * 将配置中的配置信息进行组装，校验
 * 提供刷新方法，当配置变更时，同步更新
 *
 * @author liujia
 * @date 2020/5/29 11:33
 **/
@Slf4j
public class PaymentMchConfigContext {

    private final PaymentProperties properties;

    /**
     * 支付宝支付客户端的映射关系
     * key：支付场景
     * value：支付客户端实例
     */
    private Map<String, AlipayClient> alipayClientMap = Maps.newHashMap();

    /**
     * 微信支付服务与业务场景的映射关系
     * key ： 支付场景
     * value ： 微信支付服务
     */
    private Map<String, WxPayService> wxPayServiceMap = Maps.newHashMap();

    PaymentMchConfigContext(PaymentProperties properties) {
        this.properties = properties;
        refresh(false);
    }

    /**
     * 根据场景值获取支付宝支付服务实例
     *
     * @param scene 场景值
     * @return 支付宝支付实例
     */
    public AlipayClient obtainAlipayClient(String scene) {
        AlipayClient alipayClient = alipayClientMap.get(scene);

        if (null == alipayClient) {
            log.error("请求的场景[{}]不存在对应的支付宝商户号", scene);
        }

        return alipayClient;
    }


    /**
     * 根据APPID获取支付宝账号
     *
     * @param appId 支付宝应用ID
     * @return 支付宝账号信息
     */
    public AlipayMchAccount obtainAlipayAccount(String appId) {

        for (Map.Entry<String, AlipayMchAccount> entry : properties.getAlipayMchAccounts().entrySet()) {
            if (StringUtils.equals(appId, entry.getValue().getAppId())) {
                return entry.getValue();
            }
        }

        return null;
    }

    /**
     * 根据业务场景获取对应的微信支付服务实例
     *
     * @param scene 场景值
     * @return 微信支付服务实例
     */
    public WxPayService obtainWxPayService(String scene) {
        WxPayService wxPayService = wxPayServiceMap.get(scene);

        if (null == wxPayService) {
            log.error("请求的场景[{}]不存在对应的微信商户号", scene);
        } else {
            log.debug("请求的场景[{}]对应的微信商户号为[{}]", scene, wxPayService.getConfig().getMchId());
        }

        return wxPayService;
    }

    /**
     * 根据商户号获取微信支付服务实例
     *
     * @param mchId 商户ID
     * @return 微信支付服务实例
     */
    public WxPayService obtainWxPayServiceByMchId(String mchId) {
        for (Map.Entry<String, WxPayService> entry : wxPayServiceMap.entrySet()) {
            if (StringUtils.equals(mchId, entry.getValue().getConfig().getMchId())) {
                return entry.getValue();
            }
        }

        log.warn("不存在商户ID[{}]对应的服务实例", mchId);

        return null;
    }

    /**
     * 当配置信息变更时，触发构建
     * 配置信息构建成功后才替换已经存在的构建结果，防止调用异常
     */
    public void refresh(boolean reloadKeypairFile) {
        Map<String, WxPayService> swapWxpayServiceMap = Maps.newHashMap();
        Map<String, AlipayClient> swapAlipayClientMap = Maps.newHashMap();

        for (Map.Entry<String, WechatMchAccount> entry : properties.getWechatMchAccounts().entrySet()) {
            swapWxpayServiceMap.put(entry.getKey(), build(entry.getValue(), reloadKeypairFile));
        }

        for (Map.Entry<String, AlipayMchAccount> entry : properties.getAlipayMchAccounts().entrySet()) {
            AlipayClient alipayClient = build(entry.getValue());
            swapAlipayClientMap.put(entry.getKey(), alipayClient);
        }

        wxPayServiceMap = swapWxpayServiceMap;
        alipayClientMap = swapAlipayClientMap;
    }

    /**
     * 构建一个微信支付的服务接口
     *
     * @param account           配置信息中的微信支付账号信息
     * @param reloadKeypairFile 是否重新加载密钥文件
     * @return 微信支付服务接口实例
     */
    private WxPayService build(WechatMchAccount account, boolean reloadKeypairFile) {
        WxPayConfig payConfig = new WxPayConfig();
        payConfig.setMchId(account.getMchId());
        payConfig.setMchKey(account.getMchKey());
        //获取密钥文件
        payConfig.setKeyPath(get(account, reloadKeypairFile));
        // 设置回调地址
        payConfig.setNotifyUrl(account.getNotifyUrl());
        // 可以指定是否使用沙箱环境
        payConfig.setUseSandboxEnv(account.getSandbox());

        WxPayService wxPayService = new WxPayServiceImpl();
        wxPayService.setConfig(payConfig);
        if (Boolean.TRUE.equals(account.getSandbox())) {
            try {
                payConfig.setMchKey(wxPayService.getSandboxSignKey());
            } catch (WxPayException e) {
                log.error("沙箱环境开启失败：" + e.getMessage(), e);
            }
        }

        return wxPayService;
    }

    /**
     * 通过文件路径获取指定的文件
     *
     * @param account  微信支付账号信息
     * @param override 是否覆盖，当文件为网络文件时进行覆盖
     * @return 最终的文件路径
     */
    private String get(WechatMchAccount account, boolean override) {
        // 非网络文件直接返回
        if (!startsWith(account.getKeyPath(), "http")) {
            return account.getKeyPath();
        }

        File keyPairFileRoot = new File(properties.getKeypairPath());

        if (keyPairFileRoot.isFile()) {
            keyPairFileRoot = keyPairFileRoot.getParentFile();
        }

        if (!keyPairFileRoot.canWrite()) {
            log.warn("配置的密钥文件存放地址[{}]无读取权限，修改存放地址为临时目录", properties.getKeypairPath());
            keyPairFileRoot = new File(System.getProperty("user.dir"));
        }

        String finalFilePath = keyPairFileRoot.getAbsolutePath() + account.getMchId() + ".wst";

        File keyPairFile = new File(finalFilePath);

        if (override || !keyPairFile.exists()) {
            OkHttpUtils.download(account.getKeyPath(), finalFilePath);
        }

        if (!keyPairFile.exists()) {
            log.error("密钥文件下载失败，密钥文件地址为：{}", account.getKeyPath());
        }

        return finalFilePath;
    }

    /**
     * 构建支付宝的支付客户端
     *
     * @param account 配置信息中的支付宝账号信息
     * @return 支付宝支付服务接口实例
     */
    private AlipayClient build(AlipayMchAccount account) {
        return new DefaultAlipayClient(account.getGateWayUrl(),
                account.getAppId(),
                account.getPrivateKey(),
                account.getFormat(),
                account.getCharset(),
                account.getPublicKey(),
                account.getSignType());
    }
}
