package com.bxm.newidea.component.payment.service.impl;

import com.bxm.newidea.component.payment.constant.PaymentConstant;
import com.bxm.newidea.component.payment.enums.PaymentActionEnum;
import com.bxm.newidea.component.payment.enums.PaymentPlatformEnum;
import com.bxm.newidea.component.payment.request.WechatTransfersRequest;
import com.bxm.newidea.component.payment.request.WxPayWithdrawRequest;
import com.bxm.newidea.component.payment.response.WechatTransfersResponse;
import com.bxm.newidea.component.payment.response.WxWithdrawResult;
import com.bxm.newidea.component.payment.service.IPaymentAction;
import com.bxm.newidea.component.tools.StringUtils;
import com.bxm.newidea.component.wechat.config.WechatMpContext;
import com.github.binarywang.wxpay.bean.request.BaseWxPayRequest;
import com.github.binarywang.wxpay.exception.WxPayException;
import com.github.binarywang.wxpay.service.WxPayService;
import com.thoughtworks.xstream.XStream;
import lombok.extern.slf4j.Slf4j;
import me.chanjar.weixin.mp.api.WxMpService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

/**
 * 微信付款处理流程
 *
 * @author liujia
 * @date 2020-07-01 13:47
 **/
@Component
@Slf4j
public class WechatTransfersAction extends BasePaymentAction implements IPaymentAction<WechatTransfersRequest, WechatTransfersResponse> {

    @Autowired
    private WechatMpContext wechatMpContext;

    @Override
    public WechatTransfersResponse exec(WechatTransfersRequest request) {
        WxPayService wxPayService = configContext.obtainWxPayService(request.getScene());
        if (wxPayService == null) {
            return null;
        }

        WxPayWithdrawRequest wxPayWithdrawRequest = generateWithdraw(request);

        try {
            wxPayWithdrawRequest.checkAndSign(wxPayService.getConfig());
            String url = wxPayService.getPayBaseUrl() + "/mmpaymkttransfers/promotion/transfers";
            String responseContent = wxPayService.post(url, wxPayWithdrawRequest.toXML(), Boolean.TRUE);

            XStream xstream = new XStream();
            XStream.setupDefaultSecurity(xstream);
            xstream.processAnnotations(WxWithdrawResult.class);
            xstream.allowTypes(new Class[]{WxWithdrawResult.class});

            WxWithdrawResult result = (WxWithdrawResult) xstream.fromXML(responseContent);

            WechatTransfersResponse response = new WechatTransfersResponse();
            response.setResult(result);
            response.setSuccess(StringUtils.equals(PaymentConstant.WECHAT_SUCCSS_CODE, result.getResultCode())
                    && StringUtils.equals(PaymentConstant.WECHAT_SUCCSS_CODE, result.getReturnCode()));
            return response;
        } catch (WxPayException e) {
            log.error(e.getMessage(), e);
            log.error("发起提现失败，请求参数为：{}", request);
        }

        return null;
    }

    private WxPayWithdrawRequest generateWithdraw(WechatTransfersRequest request) {
        WxMpService wxMpService = wechatMpContext.obtain(request.getScene());
        String appId = wxMpService.getWxMpConfigStorage().getAppId();

        WxPayWithdrawRequest wxPayWithdrawRequest = new WxPayWithdrawRequest();

        //如果是app提现，运营后台触发的将传递null
        wxPayWithdrawRequest.setAppid(appId);
        wxPayWithdrawRequest.setOpenId(request.getWechatOpenId());
        wxPayWithdrawRequest.setAmount(BaseWxPayRequest.yuanToFen(request.getAmount().toString()));
        wxPayWithdrawRequest.setCheckName("NO_CHECK");
        wxPayWithdrawRequest.setDesc(request.getDescription());
        wxPayWithdrawRequest.setPartnerTradeNo(request.getOrderNo());
        wxPayWithdrawRequest.setSpbillCreateIp(request.getReqeusetIp());
        return wxPayWithdrawRequest;
    }

    @Override
    public PaymentPlatformEnum matchPlatform() {
        return PaymentPlatformEnum.ALIPAY;
    }

    @Override
    public PaymentActionEnum matchAction() {
        return PaymentActionEnum.TRANSFERS;
    }
}
