package com.bxm.newidea.component.tools;

import net.sourceforge.pinyin4j.PinyinHelper;
import net.sourceforge.pinyin4j.format.HanyuPinyinCaseType;
import net.sourceforge.pinyin4j.format.HanyuPinyinOutputFormat;
import net.sourceforge.pinyin4j.format.HanyuPinyinToneType;
import net.sourceforge.pinyin4j.format.HanyuPinyinVCharType;
import net.sourceforge.pinyin4j.format.exception.BadHanyuPinyinOutputFormatCombination;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * 汉字转拼音工具类
 */
public class PinyinUtils {

    private static final Logger LOGGER = LoggerFactory.getLogger(PinyinUtils.class);

    private PinyinUtils() {
    }

    /**
     * 将汉字转换为全拼
     * @param src
     * @return String
     */
    public static String getPinYin(String src) {
        char[] t1 = src.toCharArray();
        String[] t2;
        // 设置汉字拼音输出的格式
        HanyuPinyinOutputFormat t3 = new HanyuPinyinOutputFormat();
        t3.setCaseType(HanyuPinyinCaseType.LOWERCASE);
        t3.setToneType(HanyuPinyinToneType.WITHOUT_TONE);
        t3.setVCharType(HanyuPinyinVCharType.WITH_V);
        StringBuilder t4 = new StringBuilder();
        int t0 = t1.length;
        try {
            for (char aT1 : t1) {
                // 判断是否为汉字字符
                if (Character.toString(aT1).matches("[\\u4E00-\\u9FA5]+")) {
                    // 将汉字的几种全拼都存到t2数组中
                    t2 = PinyinHelper.toHanyuPinyinStringArray(aT1, t3);
                    // 取出该汉字全拼的第一种读音并连接到字符串t4后
                    t4.append(t2[0]);
                } else {
                    // 如果不是汉字字符，直接取出字符并连接到字符串t4后
                    t4.append(Character.toString(aT1));
                }
            }
        } catch (BadHanyuPinyinOutputFormatCombination e) {
            LOGGER.error(e.getMessage(), e);
        }
        return t4.toString();
    }

    /**
     * 将字符串转换为拼音数组
     * @param src
     * @return
     */
    public static String[] getPinyinArray(String src) {
        char[] t1 = src.toCharArray();
        String[] t2;
        String[] res = new String[t1.length];
        // 设置汉字拼音输出的格式
        HanyuPinyinOutputFormat t3 = new HanyuPinyinOutputFormat();
        t3.setCaseType(HanyuPinyinCaseType.LOWERCASE);
        t3.setToneType(HanyuPinyinToneType.WITHOUT_TONE);
        t3.setVCharType(HanyuPinyinVCharType.WITH_V);

        try {
            int index = 0;
            for (char aT1 : t1) {
                // 判断是否为汉字字符
                if (Character.toString(aT1).matches("[\\u4E00-\\u9FA5]+")) {
                    // 将汉字的几种全拼都存到t2数组中
                    t2 = PinyinHelper.toHanyuPinyinStringArray(aT1, t3);
                    // 取出该汉字全拼的第一种读音并连接到字符串t4后
                    res[index++] = t2[0];
                } else {
                    // 如果不是汉字字符，直接取出字符并连接到字符串t4后
                    res[index++] = Character.toString(aT1);
                }
            }
        } catch (BadHanyuPinyinOutputFormatCombination e) {
            LOGGER.error(e.getMessage(), e);
        }
        return res;
    }

    /**
     * 提取每个汉字的首字母
     * @param str 汉字字符串
     * @return String
     */
    public static String getPinYinHeadChar(String str) {
        StringBuilder convert = new StringBuilder();
        for (int j = 0; j < str.length(); j++) {
            char word = str.charAt(j);
            // 提取汉字的首字母
            String[] pinyinArray = PinyinHelper.toHanyuPinyinStringArray(word);
            if (pinyinArray != null) {
                convert.append(pinyinArray[0].charAt(0));
            } else {
                convert.append(word);
            }
        }
        return convert.toString();
    }

    /**
     * 将字符串转换成ASCII码
     * @param cnStr 字符串
     * @return String
     */
    public static String getCnASCII(String cnStr) {
        StringBuilder strBuf = new StringBuilder();
        // 将字符串转换成字节序列
        byte[] bGBK = cnStr.getBytes();
        for (byte aBGBK : bGBK) {
            // 将每个字符转换成ASCII码
            strBuf.append(Integer.toHexString(aBGBK & 0xff)).append(" ");
        }
        return strBuf.toString();
    }

}