package com.bxm.component.poster.template.element;

import com.bxm.component.poster.context.PosterContext;
import com.bxm.component.poster.utils.FileLoadUtil;
import com.thoughtworks.xstream.annotations.XStreamAsAttribute;
import lombok.Data;
import lombok.EqualsAndHashCode;
import lombok.extern.slf4j.Slf4j;

import javax.imageio.ImageIO;
import java.awt.*;
import java.awt.geom.RoundRectangle2D;
import java.awt.image.BufferedImage;
import java.io.IOException;
import java.io.InputStream;

/**
 * 图片元素渲染
 *
 * @author liujia
 * @date 12/17/20 10:55 AM
 **/
@EqualsAndHashCode(callSuper = true)
@Slf4j
@Data
public class ImageElement extends AbstractElement {

    /**
     * 图像本身的宽度
     * required : true
     */
    @XStreamAsAttribute
    private int width;

    /**
     * 图像本身的高度
     * required : true
     */
    @XStreamAsAttribute
    private int height;

    /**
     * 圆角半径，如果提供，则图片显示为圆形
     */
    @XStreamAsAttribute
    private Integer cornerRadius;

    @Override
    public void render(Graphics2D posterGraphics2D, PosterContext context) throws IOException {
        String imagePath = context.getValue(getField());
        InputStream inputStream = FileLoadUtil.load(imagePath, true);

        if (null == inputStream) {
            log.error("渲染海报提供的图片地址[{}]不存在", imagePath);
            return;
        }
        BufferedImage subImage = ImageIO.read(inputStream);

        if (cornerRadius != null) {
            subImage = makeRoundedCorner(subImage, cornerRadius);
        }
        posterGraphics2D.drawImage(subImage, getX(), getY(), width, height, null);
    }

    /**
     * 将图片转化为圆角图片
     *
     * @param image        叠加的图片
     * @param cornerRadius 圆角半径
     * @return 处理后的圆角图片
     */
    private BufferedImage makeRoundedCorner(BufferedImage image, int cornerRadius) {
        int cornerWidth = image.getWidth();
        int cornerHeight = image.getHeight();
        BufferedImage output = new BufferedImage(cornerWidth, cornerHeight, BufferedImage.TYPE_INT_ARGB);
        Graphics2D g2 = output.createGraphics();
        g2.setComposite(AlphaComposite.Src);
        g2.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
        g2.setColor(Color.WHITE);
        g2.fill(new RoundRectangle2D.Float(0, 0, cornerWidth, cornerHeight, cornerRadius, cornerRadius));
        g2.setComposite(AlphaComposite.SrcAtop);
        g2.drawImage(image, 0, 0, null);
        g2.dispose();
        return output;
    }
}
