package com.bxm.component.poster.template.element;

import com.bxm.component.poster.context.PosterContext;
import com.bxm.component.poster.template.extend.FontStyle;
import com.bxm.component.poster.utils.FontManage;
import com.bxm.newidea.component.tools.StringUtils;
import com.thoughtworks.xstream.annotations.XStreamAlias;
import com.thoughtworks.xstream.annotations.XStreamAsAttribute;
import lombok.Data;
import lombok.EqualsAndHashCode;

import java.awt.*;
import java.awt.font.TextAttribute;
import java.text.AttributedString;
import java.util.ArrayList;
import java.util.List;

import static org.apache.commons.lang3.math.NumberUtils.toInt;

/**
 * 文本类型的渲染元素
 *
 * @author liujia
 * @date 12/17/20 10:53 AM
 **/
@EqualsAndHashCode(callSuper = true)
@Data
public class TextElement extends AbstractElement {

    /**
     * 定义文字的字体
     */
    @XStreamAlias("font")
    private FontStyle fontStyle;

    /**
     * 文本内容最大长度
     */
    @XStreamAsAttribute
    private Integer max;

    /**
     * 超长部分替换符号
     */
    @XStreamAsAttribute
    private String replace;

    /**
     * 字体颜色，rgb颜色，3个数字拼接
     */
    @XStreamAsAttribute
    private String color;

    /**
     * 是否限定文字宽度，超过宽度后将自动换行
     */
    @XStreamAsAttribute
    private Integer width;

    @Override
    public void render(Graphics2D posterGraphics2D, PosterContext context) {

        // 对内容进行处理
        String content = context.getValue(getField());

        if (null != max && content.length() > getMax()) {
            content = StringUtils.substring(content, 0, getMax());
            if (null != replace) {
                content += replace;
            }
        }

        setColor(posterGraphics2D);

        if (null == fontStyle) {
            fontStyle = new FontStyle();
            fontStyle.setStyle(Font.PLAIN);
            fontStyle.setSize(20);
        }
        Font font = FontManage.loadFont(fontStyle);

        posterGraphics2D.setFont(font);

        FontMetrics fontMetrics = posterGraphics2D.getFontMetrics(font);

        // 如果指定了宽度，则进行折行处理
        if (null != width) {
            drawString(posterGraphics2D, fontMetrics, font, content);
        } else {
            posterGraphics2D.drawString(attributedString(content, font).getIterator(),
                    getX(),
                    getY() + fontMetrics.getAscent());
        }
    }

    private AttributedString attributedString(String content, Font font) {
        AttributedString attributedString = new AttributedString(content);
        attributedString.addAttribute(TextAttribute.FONT, font);

        fillTextAttribute(attributedString);
        return attributedString;
    }

    /**
     * 填充字体显示的额外熟悉
     *
     * @param attributedString 字体熟悉设置
     */
    private void fillTextAttribute(AttributedString attributedString) {
        if (fontStyle == null) {
            return;
        }

        if (null != fontStyle.getLine()) {
            if ("STRIKETHROUGH".equalsIgnoreCase(fontStyle.getLine())) {
                attributedString.addAttribute(TextAttribute.STRIKETHROUGH, TextAttribute.STRIKETHROUGH_ON);
            } else if ("UNDERLINE".equalsIgnoreCase(fontStyle.getLine())) {
                attributedString.addAttribute(TextAttribute.UNDERLINE, TextAttribute.UNDERLINE_ON);
            }
        }

        if (null == fontStyle.getWeight() || "SEMIBOLD".equalsIgnoreCase(fontStyle.getWeight())) {
            attributedString.addAttribute(TextAttribute.WEIGHT, TextAttribute.WEIGHT_SEMIBOLD);
        }
    }

    /**
     * 设置文本字体颜色，默认设置为黑色
     *
     * @param posterGraphics2D 绘图对象
     */
    private void setColor(Graphics2D posterGraphics2D) {
        Color hsbColor = Color.BLACK;

        if (null != color) {
            String[] rgbColorArray = StringUtils.split(color, ",");
            if (rgbColorArray.length == 3) {
                hsbColor = new Color(toInt(rgbColorArray[0]), toInt(rgbColorArray[1]), toInt(rgbColorArray[2]));
            }
        }

        posterGraphics2D.setColor(hsbColor);
    }

    /**
     * 文字超出限定长度自动换行
     *
     * @param graphics2D 画布
     * @param text       文字
     */
    private void drawString(Graphics2D graphics2D, FontMetrics fontMetrics, Font font, String text) {
        int x = getX();
        int y = getY();
        int widthLength = getWidth();
        int yn = fontMetrics.getHeight();

        List<String> contentCharList = new ArrayList<>(text.length());

        getListText(fontMetrics, text, widthLength, contentCharList);

        for (int i = 0; i < contentCharList.size(); i++) {
            AttributedString attributedString = attributedString(contentCharList.get(i), font);

            graphics2D.drawString(attributedString.getIterator(),
                    x,
                    y + fontMetrics.getAscent() + i * yn);
        }
    }

    /**
     * 递归 切割字符串
     */
    private void getListText(FontMetrics fontMetrics, String text, int widthLength, List<String> contentCharList) {
        String subStr = text;
        boolean breakFlag = true;
        int i = 1;

        while (breakFlag) {
            if (fontMetrics.stringWidth(text) > widthLength) {
                text = text.substring(0, text.length() - 1);
                i++;
            } else {
                breakFlag = false;
            }
        }

        if (i != 1) {
            contentCharList.add(subStr.substring(0, subStr.length() - i));
            getListText(fontMetrics, subStr.substring(subStr.length() - i), widthLength, contentCharList);
        } else {
            contentCharList.add(text);
        }
    }
}
























