package com.bxm.component.poster.engine;

import com.bxm.component.poster.config.PosterConfiguration;
import com.bxm.component.poster.context.PosterContext;
import com.bxm.component.poster.output.IPosterOutput;
import com.bxm.component.poster.output.impl.LocalPosterOutput;
import com.bxm.component.poster.template.PosterTemplate;
import com.bxm.component.poster.utils.FontManage;
import com.bxm.newidea.component.tools.StringUtils;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.io.File;
import java.util.Map;
import java.util.concurrent.atomic.AtomicBoolean;

/**
 * 海报生成引擎
 *
 * @author liujia
 * @date 12/17/20 11:59 AM
 **/
@Component
@Slf4j
public class PosterEngine {

    /**
     * 默认的输出策略
     */
    private IPosterOutput defaultPosterOutput = new LocalPosterOutput();

    private final PosterConfiguration posterConfiguration;

    private AtomicBoolean initFlag = new AtomicBoolean(false);

    @Autowired
    public PosterEngine(PosterConfiguration posterConfiguration) {
        this.posterConfiguration = posterConfiguration;
    }

    /**
     * 通过参数上下文生成海报
     * 海报模板通过xml文件进行配置，海报需要的相关信息通过参数进行传递
     * 组件本身不对业务逻辑进行处理
     *
     * @param context 生成海报的相关参数和定制化参数
     * @return 生成后的海报存放地址，根据选取的输出策略会有区别，可能是本地路径或者远程地址
     */
    public String generate(PosterContext context) {
        long start = System.currentTimeMillis();

        init();

        if (log.isDebugEnabled()) {
            log.debug("开始创建海报，创建参数：{}", context);
        }

        // 根据策略进行输出
        IPosterOutput posterOutput = context.getPosterOutput();
        if (posterOutput == null) {
            posterOutput = defaultPosterOutput;
        }

        // 如果海报生成策略不是覆盖，则判断是否已经存在海报，存在的情况下直接返回
        if (!context.isOverwrite()) {
            String posterPath = posterOutput.exists(context);
            if (StringUtils.isNotBlank(posterPath)) {
                return posterPath;
            }
        }

        // 获取模板
        PosterTemplate template = loadTemplate(context);
        if (template == null) {
            log.error("不存在[{}]对应的模板，请检查配置", context.getTemplate());
            return StringUtils.EMPTY;
        }

        // 渲染模板
        File tempPosterFile = null;
        try {
            tempPosterFile = template.render(context);
            String output = posterOutput.output(context, tempPosterFile);

            if (log.isDebugEnabled()) {
                log.debug("海报合成结束，总耗时：{}", System.currentTimeMillis() - start);
            }
            return output;
        } finally {
            if (null != tempPosterFile && tempPosterFile.exists()) {
                if (!tempPosterFile.delete()) {
                    log.warn("文件未正常移除，文件路径：{}", tempPosterFile.getAbsolutePath());
                }
            }
        }
    }

    private void init() {
        if (initFlag.compareAndSet(false, true)) {
            FontManage.init();
        }
    }

    private PosterTemplate loadTemplate(PosterContext context) {
        for (Map.Entry<String, PosterTemplate> entry : posterConfiguration.getTemplateMap().entrySet()) {
            if (StringUtils.equals(context.getTemplate(), entry.getValue().getName())) {
                return entry.getValue();
            }

        }
        return null;
    }
}
