package com.bxm.component.poster.template.element;

import com.bxm.component.poster.context.PosterContext;
import com.bxm.component.poster.template.extend.LogoImage;
import com.bxm.component.poster.utils.FileLoadUtil;
import com.bxm.newidea.component.CodeCreaterFactory;
import com.bxm.newidea.component.config.ImageTypeEnum;
import com.bxm.newidea.component.param.QrCodeParam;
import com.bxm.newidea.component.tools.FileUtils;
import com.thoughtworks.xstream.annotations.XStreamAlias;
import com.thoughtworks.xstream.annotations.XStreamAsAttribute;
import lombok.Data;
import lombok.EqualsAndHashCode;
import lombok.extern.slf4j.Slf4j;

import javax.imageio.ImageIO;
import java.awt.*;
import java.awt.image.BufferedImage;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.util.UUID;

import static org.apache.commons.io.FileUtils.forceDelete;

/**
 * 将二维码绘制在画布上，可以提供二维码内容，也可以是已经生成的二维码图片地址
 *
 * @author liujia
 * @date 12/17/20 10:55 AM
 **/
@EqualsAndHashCode(callSuper = true)
@Data
@Slf4j
public class QrcodeElement extends ImageElement {

    /**
     * 是否需要生成二维码，true表示直接使用内容作为图片追加，false表示将内容生成二维码
     */
    @XStreamAsAttribute
    private Boolean completed;

    /**
     * 二维码中心的logo，completed为false，并且提供了的情况，会使用此图片的地址作为二维码的中心logo
     */
    @XStreamAlias("logo")
    private LogoImage logo;

    @Override
    public void render(Graphics2D posterGraphics2D, PosterContext context) throws IOException {
        String source = context.getValue(getField());

        BufferedImage qrcodeImg;
        File qrcodeFile = null;

        try {
            if (completed) {
                InputStream inputStream = FileLoadUtil.load(source, true);
                if (null == inputStream) {
                    log.error("渲染海报提供的二维码图片地址[{}]不存在", source);
                    return;
                }
                qrcodeImg = ImageIO.read(inputStream);
            } else {
                qrcodeFile = new File(tempQrcodeFilePath());
                qrcodeImg = createQrcode(source, qrcodeFile);
            }

            posterGraphics2D.drawImage(qrcodeImg, getX(), getY(), getWidth(), getHeight(), null);
        } finally {
            if (null != qrcodeFile) {
                forceDelete(qrcodeFile);
            }
        }
    }

    private String tempQrcodeFilePath() {
        return FileUtils.getClassRootPath() + "temp" + File.separator +
                UUID.randomUUID().toString() + ".jpg";
    }

    private BufferedImage createQrcode(String source, File qrcodeFile) throws IOException {
        QrCodeParam param = new QrCodeParam(qrcodeFile, source);

        if (null != logo && null != logo.getUrl()) {
            File logoFile = FileLoadUtil.loadFile(logo.getUrl(), true);
            if (logoFile != null) {
                param.setLogoPath(logoFile.getAbsolutePath());
            }
        }

        param.setHeight(getWidth());
        param.setWidth(getHeight());
        param.setImageTypeEnum(ImageTypeEnum.JPEG);
        CodeCreaterFactory.createQrCode(param);

        return ImageIO.read(qrcodeFile);
    }
}
