package com.bxm.component.poster.utils;

import cn.hutool.core.date.DateUtil;
import cn.hutool.core.date.TimeInterval;
import cn.hutool.http.HttpUtil;
import com.bxm.newidea.component.tools.StringUtils;
import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;

import java.io.BufferedInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.InputStream;

import static com.bxm.newidea.component.tools.FileUtils.getFilextension;

/**
 * @author liujia
 * @date 12/17/20 2:36 PM
 **/
@Slf4j
public class FileLoadUtil {

    /**
     * 加载文件，支持加载本地文件与远程文件
     *
     * @param filePath 文件路径，本地路径或者远程网络地址
     * @param cache    如果是远程网络文件，是否缓存到本地
     * @return 文件输入流
     */
    @SneakyThrows
    public static InputStream load(String filePath, boolean cache) {
        File loadFile = loadFile(filePath, cache);

        if (null != loadFile) {
            if (!loadFile.canRead()) {
                log.error("文件[{}]不可读取，请检查权限", loadFile.getAbsolutePath());
                return null;
            }

            if (!loadFile.exists()) {
                log.error("文件[{}]不存在", loadFile.getAbsolutePath());
                return null;
            }

            return new BufferedInputStream(new FileInputStream(loadFile));
        }

        return null;
    }

    private static String buildLocalPath(String remotePath) {
        String tempDir = System.getProperty("temp.dir");
        if (null == tempDir) {
            tempDir = System.getProperty("java.io.tmpdir");
            if (StringUtils.isBlank(tempDir)) {
                tempDir = "/opt/temp";
            }
        }
        //生成临时文件 并获取目录
        File sysTempDir = new File(tempDir);

        if (!sysTempDir.exists()) {
            boolean mkdir = sysTempDir.mkdirs();
            if (!mkdir) {
                log.error("创建临时目录失败，请检查权限问题，目录：{}", sysTempDir);
            }
        }

        if (!tempDir.endsWith("/")) {
            tempDir += "/";
        }

        long hashCode = remotePath.hashCode() & Integer.MAX_VALUE;

        String fileExtension = getFilextension(remotePath);
        String finalFilePath = tempDir + "d" + (hashCode % 100) + File.separator + "f" + hashCode + "." + fileExtension;

        if (log.isDebugEnabled()) {
            log.debug("文件：{}下载后的本地存储地址：{}", remotePath, finalFilePath);
        }

        return finalFilePath;
    }

    private static boolean isRemote(String filePath) {
        return StringUtils.startsWithAny(filePath, "http:", "https:");
    }

    /**
     * 加载本地文件
     *
     * @param filePath 文件路径，本地路径或者远程网络地址
     * @param cache    如果是远程网络文件，是否缓存到本地
     * @return 文件地址对应的文件
     */
    public static File loadFile(String filePath, boolean cache) {
        File localFile;

        if (isRemote(filePath)) {
            String localPath = buildLocalPath(filePath);

            localFile = new File(localPath);

            if (cache && localFile.exists()) {
                log.info("加载本地缓存文件：{}", localPath);
                return localFile;
            }

            TimeInterval timer = DateUtil.timer();
            HttpUtil.downloadFile(filePath, localFile);
            if (log.isDebugEnabled()) {
                log.debug("下载耗时：{}", timer.intervalMs());
            }
        } else {
            localFile = new File(filePath);

            if (!localFile.exists()) {
                return null;
            }

        }
        return localFile;
    }
}