package com.bxm.component.poster.template;

import com.bxm.component.poster.context.PosterContext;
import com.bxm.component.poster.template.element.*;
import com.bxm.component.poster.utils.FileLoadUtil;
import com.bxm.component.poster.utils.FontManage;
import com.bxm.newidea.component.tools.DateUtils;
import com.bxm.newidea.component.tools.FileUtils;
import com.google.common.collect.Lists;
import com.thoughtworks.xstream.annotations.XStreamAlias;
import com.thoughtworks.xstream.annotations.XStreamAsAttribute;
import com.thoughtworks.xstream.annotations.XStreamImplicit;
import lombok.Data;
import lombok.ToString;
import lombok.extern.slf4j.Slf4j;

import javax.imageio.ImageIO;
import java.awt.*;
import java.awt.image.BufferedImage;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.util.Date;
import java.util.List;

/**
 * 海报模板定义，支持外部配置解析
 *
 * @author liujia
 * @date 12/17/20 11:05 AM
 **/
@Data
@Slf4j
@XStreamAlias(value = "template")
@ToString
public class PosterTemplate {

    /**
     * 渲染海报，将海报配置的子元素均叠加到海报上
     * 元素具体如何渲染、位置、样式等均由配置文件进行控制
     *
     * @param context 海报渲染上下文
     * @return 海报临时文件（写入临时文件效率高于写入buffer流）
     */
    public File render(PosterContext context) {
        // 初始化字体文件
        FontManage.init();
        
        String loadIamge = backgroundImage;

        if (null != context.getMainImage()) {
            loadIamge = context.getMainImage();
        }

        try (InputStream inputStream = FileLoadUtil.load(loadIamge, true)) {
            if (inputStream == null) {
                return null;
            }

            BufferedImage backgroundImageBuffered = ImageIO.read(inputStream);
            Graphics2D posterGraphics2D = backgroundImageBuffered.createGraphics();
            // 解决文字锯齿
            posterGraphics2D.setRenderingHint(
                RenderingHints.KEY_TEXT_ANTIALIASING, RenderingHints.VALUE_TEXT_ANTIALIAS_LCD_HRGB);

            for (AbstractElement element : getAllElements()) {
                element.render(posterGraphics2D, context);
            }

            posterGraphics2D.dispose();

            String tempImgFileName = FileUtils.getClassRootPath() + File.separator +
                "temp/" + DateUtils.PATTERN_NO_DELIMITER_FORMAT.get().format(new Date()) +
                "/" + context.hashCode() + "." + context.getFileExtension();

            File tempImgFile = new File(tempImgFileName);
            FileUtils.checkAndCreateFile(tempImgFile);

            ImageIO.write(backgroundImageBuffered, context.getFileExtension(), tempImgFile);

            return tempImgFile;
        } catch (IOException e) {
            log.error(e.getMessage(), e);
            log.error("文件读取失败，请求参数：{},当前模板配置：{}", context, this);
        }
        return null;
    }

    private List<AbstractElement> getAllElements() {
        if (elements == null) {
            elements = Lists.newArrayList();

            if (null != textElements) {
                elements.addAll(textElements);
            }
            if (null != qrcodeElements) {
                elements.addAll(qrcodeElements);
            }
            if (null != imageElements) {
                elements.addAll(imageElements);
            }
            if (null != circularImageElements) {
                elements.addAll(circularImageElements);
            }
        }
        return elements;
    }

    private List<AbstractElement> elements;

    /**
     * 模板名称，用于和需要生成的模板名称匹配
     */
    @XStreamAsAttribute
    private String name;

    /**
     * 模板的背景底图地址，图片地址会缓存到本地
     */
    @XStreamAsAttribute
    private String backgroundImage;

    /**
     * 渲染到底图上的文本元素列表
     */
    @XStreamImplicit(itemFieldName = "text")
    private List<TextElement> textElements = Lists.newArrayList();

    /**
     * 渲染到底图上的二维码元素列表
     */
    @XStreamImplicit(itemFieldName = "qrcode")
    private List<QrcodeElement> qrcodeElements = Lists.newArrayList();

    /**
     * 渲染到底图上的图片元素列表
     */
    @XStreamImplicit(itemFieldName = "image")
    private List<ImageElement> imageElements = Lists.newArrayList();

    /**
     * 圆形图片元素
     */
    @XStreamImplicit(itemFieldName = "circular")
    private List<CircularImageElement> circularImageElements = Lists.newArrayList();
}
