package com.bxm.component.poster.template.element;

import com.bxm.component.poster.context.PosterContext;
import com.bxm.component.poster.template.extend.FontStyle;
import com.bxm.component.poster.utils.FontManage;
import com.bxm.newidea.component.tools.StringUtils;
import com.thoughtworks.xstream.annotations.XStreamAlias;
import com.thoughtworks.xstream.annotations.XStreamAsAttribute;
import lombok.Data;
import lombok.EqualsAndHashCode;

import java.awt.*;
import java.awt.font.TextAttribute;
import java.text.AttributedString;
import java.util.ArrayList;
import java.util.List;

import static org.apache.commons.lang3.math.NumberUtils.toInt;

/**
 * 文本类型的渲染元素
 *
 * @author liujia
 * @date 12/17/20 10:53 AM
 **/
@EqualsAndHashCode(callSuper = true)
@Data
public class TextElement extends AbstractElement {

    /**
     * 定义文字的字体
     */
    @XStreamAlias("font")
    private FontStyle fontStyle;

    /**
     * 文本内容最大长度
     */
    @XStreamAsAttribute
    private Integer max;

    /**
     * 超长部分替换符号
     */
    @XStreamAsAttribute
    private String replace;

    /**
     * 字体颜色，rgb颜色，3个数字拼接
     */
    @XStreamAsAttribute
    private String color;

    /**
     * 是否限定文字宽度，超过宽度后将自动换行
     */
    @XStreamAsAttribute
    private Integer width;

    /**
     * 文字对齐方式，支持：RIGHT、CENTER
     * 如果配置居中对齐或者右边对齐，则需要提供右侧的x坐标
     */
    @XStreamAsAttribute
    private String align;

    /**
     * 右侧位置的x坐标，用于计算文字居中或居右对齐的位置
     */
    @XStreamAsAttribute
    private Integer rightX;

    @Override
    public void render(Graphics2D posterGraphics2D, PosterContext context) {

        // 对内容进行处理
        String content = context.getValue(getField());

        if (null != max && content.length() > getMax()) {
            content = StringUtils.substring(content, 0, getMax());
            if (null != replace) {
                content += replace;
            }
        }

        if (content.length() == 0) {
            return;
        }

        setColor(posterGraphics2D);

        if (null == fontStyle) {
            fontStyle = new FontStyle();
            fontStyle.setStyle(Font.PLAIN);
            fontStyle.setSize(20);
        }
        Font font = FontManage.loadFont(fontStyle);

        posterGraphics2D.setFont(font);

        FontMetrics fontMetrics = posterGraphics2D.getFontMetrics(font);

        // 如果指定了宽度，则进行折行处理
        if (null != width) {
            drawMultiString(posterGraphics2D, fontMetrics, font, content);
        } else {
            drawString(posterGraphics2D, content, font, fontMetrics, 0);
        }
    }

    private void drawString(Graphics2D posterGraphics2D, String content, Font font, FontMetrics fontMetrics, int extHeight) {
        int floatXPoint = getX();

        if (align != null && rightX != null) {
            int innerWidth = rightX - getX();
            int contentWidth = fontMetrics.stringWidth(content);

            // 如果是居中对齐，调整左边的文字起点
            if ("CENTER".equalsIgnoreCase(align)) {
                if (contentWidth < innerWidth) {
                    floatXPoint += (innerWidth - contentWidth) / 2;
                }
            } else if ("RIGHT".equalsIgnoreCase(align)) {
                floatXPoint += innerWidth - contentWidth;
            }

        }

        posterGraphics2D.drawString(attributedString(content, font).getIterator(),
                floatXPoint,
                getY() + fontMetrics.getAscent() + extHeight);
    }

    private AttributedString attributedString(String content, Font font) {
        AttributedString attributedString = new AttributedString(content);
        attributedString.addAttribute(TextAttribute.FONT, font);

        fillTextAttribute(attributedString);
        return attributedString;
    }

    /**
     * 填充字体显示的额外熟悉
     *
     * @param attributedString 字体熟悉设置
     */
    private void fillTextAttribute(AttributedString attributedString) {
        if (fontStyle == null) {
            return;
        }

        if (null != fontStyle.getLine()) {
            if ("STRIKETHROUGH".equalsIgnoreCase(fontStyle.getLine())) {
                attributedString.addAttribute(TextAttribute.STRIKETHROUGH, TextAttribute.STRIKETHROUGH_ON);
            } else if ("UNDERLINE".equalsIgnoreCase(fontStyle.getLine())) {
                attributedString.addAttribute(TextAttribute.UNDERLINE, TextAttribute.UNDERLINE_ON);
            }
        }

        if (null == fontStyle.getWeight() || "SEMIBOLD".equalsIgnoreCase(fontStyle.getWeight())) {
            attributedString.addAttribute(TextAttribute.WEIGHT, TextAttribute.WEIGHT_SEMIBOLD);
        }
    }

    /**
     * 设置文本字体颜色，默认设置为黑色
     *
     * @param posterGraphics2D 绘图对象
     */
    private void setColor(Graphics2D posterGraphics2D) {
        Color hsbColor = Color.BLACK;

        if (null != color) {
            String[] rgbColorArray = StringUtils.split(color, ",");
            if (rgbColorArray.length == 3) {
                hsbColor = new Color(toInt(rgbColorArray[0]), toInt(rgbColorArray[1]), toInt(rgbColorArray[2]));
            }
        }

        posterGraphics2D.setColor(hsbColor);
    }

    /**
     * 文字超出限定长度自动换行
     *
     * @param graphics2D 画布
     * @param text       文字
     */
    private void drawMultiString(Graphics2D graphics2D, FontMetrics fontMetrics, Font font, String text) {
        int x = getX();
        int y = getY();
        int widthLength = getWidth();
        int yn = fontMetrics.getHeight();

        List<String> contentCharList = new ArrayList<>(text.length());

        getListText(fontMetrics, text, widthLength, contentCharList);

        for (int i = 0; i < contentCharList.size(); i++) {
            drawString(graphics2D, contentCharList.get(i), font, fontMetrics, i * yn);
        }
    }

    /**
     * 递归 切割字符串
     */
    private void getListText(FontMetrics fontMetrics, String text, int widthLength, List<String> contentCharList) {
        String subStr = text;
        boolean breakFlag = true;
        int i = 1;

        while (breakFlag) {
            if (fontMetrics.stringWidth(text) > widthLength) {
                text = text.substring(0, text.length() - 1);
                i++;
            } else {
                breakFlag = false;
            }
        }

        if (i != 1) {
            contentCharList.add(subStr.substring(0, subStr.length() - i));
            getListText(fontMetrics, subStr.substring(subStr.length() - i), widthLength, contentCharList);
        } else {
            contentCharList.add(text);
        }
    }
}
























