package com.bxm.newidea.cloud.ribbon;

import com.bxm.newidea.cloud.constant.EnvConstant;
import com.google.common.base.Optional;
import com.netflix.loadbalancer.ILoadBalancer;
import com.netflix.loadbalancer.Server;
import com.netflix.loadbalancer.ZoneAvoidanceRule;
import com.netflix.niws.loadbalancer.DiscoveryEnabledServer;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

/**
 * 不同环境的负载规则分发
 * @author liujia
 */
public class EnvDiscoveryEnabledRule extends ZoneAvoidanceRule {

    private static final Logger LOGGER = LoggerFactory.getLogger(EnvDiscoveryEnabledRule.class);

    public EnvDiscoveryEnabledRule() {
        super();
    }

    @Override
    public Server choose(Object key) {
        final RibbonFilterContext context = RibbonFilterContextHolder.getCurrentContext();
        ILoadBalancer lb = getLoadBalancer();
        final List<Server> allServers = lb.getAllServers();
        // 匹配成功的server
        final List<Server> matchedMetaServers = new ArrayList<>();

        final Map<String, String> attributes = context.getAttributes();
        // 取得接口端传入的参数
        final String ribbonVal = attributes.get(EnvConstant.ENV_HEADER_KEY);

        LOGGER.debug("request server:" + key + ",ribbonVal:" + ribbonVal);

        if (StringUtils.isNotBlank(ribbonVal)) {
            for (Server server : allServers) {
                if (server instanceof DiscoveryEnabledServer) {
                    final DiscoveryEnabledServer discoveryEnabledServer = (DiscoveryEnabledServer) server;
                    final Map<String, String> metadata = discoveryEnabledServer.getInstanceInfo().getMetadata();
                    final String env = metadata.get(EnvConstant.ENV_HEADER_KEY);
                    // 获取符合条件的服务
                    if (StringUtils.isNotBlank(env) && EnvConstant.PRE_ENV_VALUE.equals(env)) {
                        matchedMetaServers.add(server);
                    }
                }
            }

            //优先走自定义路由。即满足灰度要求的server
            if (!matchedMetaServers.isEmpty()) {
                Optional<Server> server = getPredicate().chooseRoundRobinAfterFiltering(matchedMetaServers, key);
                if (server.isPresent()) {
                    LOGGER.debug("choose server: " + server.get().getId() + ",env:" + ribbonVal + ",key:" + key);
                    return server.get();
                } else {
                    LOGGER.error("请求头信息中包含预发键值，但是未找到符合条件的服务，请求key：" + key);
                    return null;
                }
            }
            LOGGER.error("请求头信息中包含预发键值，但是未找到对应配置的服务，请检查服务部署的eureka metadata，请求key：{},ribbonVal:{}"
                    , key
                    , ribbonVal);
            //return null;
        }

        //不存在预发键值，进行默认的路由
        Optional<Server> server = getPredicate().chooseRoundRobinAfterFiltering(allServers, key);
        if (server.isPresent()) {
            LOGGER.debug("choose server: " + server.get().getId() + ",key:" + key + ",ribbonVal:" + ribbonVal);
            return server.get();
        } else {
            return null;
        }
    }
}