/*
 * Copyright 2012-2017 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.bxm.newidea.component.quartz.config;

import com.bxm.newidea.component.quartz.AbstractCustomJob;
import org.quartz.spi.TriggerFiredBundle;
import org.springframework.beans.factory.config.AutowireCapableBeanFactory;
import org.springframework.scheduling.quartz.SpringBeanJobFactory;
import org.springframework.util.Assert;

/**
 * Subclass of {@link SpringBeanJobFactory} that supports auto-wiring job beans.
 * @author Vedran Pavic
 * @see <a href="http://blog.btmatthews.com/?p=40#comment-33797">Inject application
 * context dependencies in Quartz job beans</a>
 */
public class AutowireCapableBeanJobFactory extends SpringBeanJobFactory {

    private final AutowireCapableBeanFactory beanFactory;

    AutowireCapableBeanJobFactory(AutowireCapableBeanFactory beanFactory) {
        Assert.notNull(beanFactory, "Bean factory must not be null");
        this.beanFactory = beanFactory;
    }

    @Override
    protected Object createJobInstance(TriggerFiredBundle bundle) throws Exception {
        Object jobInstance = super.createJobInstance(bundle);
        this.beanFactory.autowireBean(jobInstance);

        if (jobInstance instanceof AbstractCustomJob) {
            AbstractCustomJob customJob = (AbstractCustomJob) jobInstance;
            if (null != bundle.getJobDetail().getJobDataMap()) {
                customJob.setParam(bundle.getJobDetail().getJobDataMap().getWrappedMap());
            }
        }
        return jobInstance;
    }

}
