package com.bxm.newidea.component.redis;

import org.springframework.data.redis.core.ZSetOperations;

import java.util.Set;

/**
 * 有序集合redis操作代理
 * @author liujia 2018/5/22 10:12
 */
public interface RedisZSetAdapter extends BaseRedisOperation{
    /**
     * 获取代理操作的元素对象
     * @return set操作类
     */
    ZSetOperations getOriginal();

    /**
     * 添加集合对象到
     * @param generator key
     * @param value     集合项
     * @param score     集合分值
     * @return 添加结果
     */
    Boolean add(KeyGenerator generator, Object value, double score);

    /**
     * 批量添加集合项到key中
     * @param generator key
     * @param tuples    集合项
     * @return 添加成功的数量
     */
    long add(KeyGenerator generator, Set<ZSetOperations.TypedTuple> tuples);

    /**
     * 给集合中的值对应的分数添加amount
     * @param generator key
     * @param value     项
     * @param amount    添加的分值
     * @return 给指定的值加上对应的分值
     */
    double incrementScore(KeyGenerator generator, Object value, double amount);

    /**
     * 从集合中移除项
     * @param generator key
     * @param values    移除的项
     * @return 移除成功的数量
     */
    Long remove(KeyGenerator generator, Object... values);

    /**
     * 移除排名在<code>start</code>到<code>end</code>之间的项
     * @param generator key
     * @param start     开始排名（起始为0）
     * @param end       截止排名
     * @return 移除的数量
     */
    Long removeByRange(KeyGenerator generator, long start, long end);

    /**
     * 移除分值在<code>min</code>到<code>max</code>之间的项
     * @param generator key
     * @param min       最小分值
     * @param max       最大分值
     * @return 移除的数量
     */
    Long removeByScore(KeyGenerator generator, double min, double max);

    /**
     * 键值对应的有序集合长度
     * @param generator key
     * @return 集合长度
     */
    Long size(KeyGenerator generator);

    /**
     * 获取集合中分值介于<code>min</code>和<code>max</code>之间的项数量
     * @param generator key
     * @param min       最低分
     * @param max       最高分
     * @return 分值之间的数量
     */
    Long count(KeyGenerator generator, double min, double max);

    /**
     * 返回value在集合中的排名
     * @param generator key
     * @param value     有序集合中的数据项
     * @return 数据项对应的排名
     */
    Long rank(KeyGenerator generator, Object value);

    /**
     * 返回value在集合中分数
     * @param generator key
     * @param value     有序集合中的数据项
     * @return 数据项的当前分数
     */
    Double score(KeyGenerator generator, Object value);

    /**
     * 根据分值进行排序，返回分值介于min/max之间并且偏移量offset之后的count条数据
     * @param generator  key
     * @param min        最小分值
     * @param max        最大分值
     * @param offset     偏移量，起始为0（类似mysql的limit）
     * @param count      查询总数
     * @param isReversed 是否反转,true表示递减，false表示递增
     * @return 查询的结果集（LinkedHashSet）,包含分值
     */
    <T> Set<ZSetOperations.TypedTuple<T>> rangeByScoreWithScores(KeyGenerator generator, double min, double max, long offset, long count, boolean isReversed);

    /**
     * 根据分值进行排序，返回分值介于min、max之间的数据
     * @param generator  key
     * @param min        最小分值
     * @param max        最大分值
     * @param isReversed 是否反转,true表示递减，false表示递增
     * @return 查询的结果集, 包含分值
     */
    <T> Set<ZSetOperations.TypedTuple<T>> rangeByScoreWithScores(KeyGenerator generator, double min, double max, boolean isReversed);

    /**
     * 排名在start到end之间的数据
     * @param generator  key
     * @param start      起始排序
     * @param end        截止排序
     * @param isReversed 是否反转,true表示递减，false表示递增
     * @return 符合排名的数据项
     */
    <T> Set<T> range(KeyGenerator generator, long start, long end, boolean isReversed);

    /**
     * 获取分值在min到max之间的数据
     * @param generator  key
     * @param min        最小分值
     * @param max        最大分值
     * @param isReversed 是否反转,true表示递减，false表示递增
     * @return 符合条件的数据项
     */
    <T> Set<T> rangeByScore(KeyGenerator generator, double min, double max, boolean isReversed);

    /**
     * 获取分值在min到max之间的数据，根据offset进行偏移，获取总共count条数据
     * @param generator  key
     * @param min        最小分值
     * @param max        最大分值
     * @param offset     偏移量
     * @param count      检索的总数
     * @param isReversed 是否反转,true表示递减，false表示递增
     * @return 符合条件的数据项
     */
    <T> Set<T> rangeByScore(KeyGenerator generator, double min, double max, long offset, long count, boolean isReversed);

    /**
     * 计算source与otherkeys的交集，并将计算结果保存到target中,相同项的分值会相加
     * @param target 存储计算结果的key
     * @param source 用于计算的原key
     * @param otherKeys 用于计算的参照key
     * @return 计算的结果数量
     */
    Long intersectAndStore(KeyGenerator target, KeyGenerator source, KeyGenerator... otherKeys);

    /**
     * 计算source与otherkeys的并集，并将计算结果保存到target中,相同项的分值会相加
     * @param target 存储计算结果的key
     * @param source 用于计算的key
     * @param otherKeys 用于计算的参照key
     * @return 计算结果数量
     */
    Long unionAndStore(KeyGenerator target,KeyGenerator source,KeyGenerator... otherKeys);
}
