package com.bxm.newidea.component.redis.impl;

import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisListAdapter;
import org.apache.commons.lang3.ArrayUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.ListOperations;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.List;

@SuppressWarnings("unchecked")
@Component
public class RedisListAdapterImpl extends BaseRedisAdapter implements RedisListAdapter {

    private ListOperations operations;

    @Autowired
    public RedisListAdapterImpl(RedisTemplate<String, Object> redisTemplate) {
        super(redisTemplate);
        operations = redisTemplate.opsForList();
    }

    @Override
    public long leftPush(KeyGenerator generator, Object... values) {
        return this.execPush(generator, true, values);
    }

    @Override
    public <T> long leftPush(KeyGenerator generator, List<T> value) {
        if(!CollectionUtils.isEmpty(value)){
            return this.execPush(generator, true, value.toArray());
        }
        return 0L;
    }

    private long execPush(KeyGenerator generator, boolean isLeft, Object[] values) {
        if (ArrayUtils.isEmpty(values)) {
            return 0L;
        }

        String key = generator.gen();

        try {
            List<byte[]> result = this.convertToByteArray(values);

            redisTemplate.watch(key);
            redisTemplate.multi();

            return isLeft ? operations.leftPushAll(key, result) : operations.rightPushAll(key, result);

        } finally {
            redisTemplate.unwatch();
        }
    }

    private List<byte[]> convertToByteArray(Object... values) {
        List<byte[]> dataArray = new ArrayList<>();

        for (Object obj : values) {
            dataArray.add(serializer.serialize(obj));
        }

        return dataArray;
    }

    @Override
    public long rightPush(KeyGenerator generator, Object... values) {
        return this.execPush(generator, false, values);
    }

    @Override
    public long size(KeyGenerator generator) {
        return operations.size(generator.gen());
    }

    @Override
    public <T> T leftPop(KeyGenerator generator) {
        return this.execPop(generator,true);
    }

    private <T> T execPop(KeyGenerator generator,boolean isLeft){
        try {
            String key = generator.gen();

            redisTemplate.watch(key);
            redisTemplate.multi();

            Object value = isLeft?operations.leftPop(generator.gen()):operations.rightPop(generator.gen());
            return this.deserialize(value);
        } finally {
            redisTemplate.unwatch();
        }
    }

    @Override
    public <T> T rightPop(KeyGenerator generator) {
        return this.execPop(generator,false);
    }

    @Override
    public <T> List<T> leftIndex(KeyGenerator generator, long index) {
        return this.range(generator,0,index);
    }

    @Override
    public <T> T index(KeyGenerator generator, long index) {
        Object value = operations.index(generator.gen(),index);
        return this.deserialize(value);
    }

    @Override
    public <T> List<T> range(KeyGenerator generator, long start, long end) {
        List<T> result = new ArrayList<>();

        List<Object> values = operations.range(generator.gen(),start,end);
        if(!CollectionUtils.isEmpty(values)){
            values.forEach(value -> result.add(this.deserialize(value)));
        }
        return result;
    }

    @Override
    public void leftTrim(KeyGenerator generator, long start, long end) {
        String key = generator.gen();
        try {
            redisTemplate.watch(key);
            redisTemplate.multi();

            operations.trim(generator.gen(),start,end);
        }finally {
            redisTemplate.unwatch();
        }
    }
}
