package com.bxm.newidea.component.redisson.impl;

import com.bxm.newidea.component.JSON;
import com.bxm.newidea.component.entity.TypedTuple;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisZSetAdapter;
import com.bxm.newidea.component.redisson.config.SwitchMultiDataSourceHandler;
import org.redisson.api.RScoredSortedSet;
import org.redisson.api.RSortedSet;
import org.redisson.client.protocol.ScoredEntry;

import java.util.*;
import java.util.concurrent.atomic.AtomicLong;

public class RedissonZSetAdapterImpl extends BaseRedisAdapter implements RedisZSetAdapter {

    public RedissonZSetAdapterImpl(SwitchMultiDataSourceHandler switchMultiDataSourceHandler) {
        super(switchMultiDataSourceHandler);
    }

    private RSortedSet zsetOperations(KeyGenerator keyGenerator) {
        return switchMultiDataSourceHandler.getRedissonClient(keyGenerator.getDbName()).getSortedSet(keyGenerator.gen());
    }

    private RScoredSortedSet scoredSortedSetOperations(KeyGenerator keyGenerator) {
        return switchMultiDataSourceHandler.getRedissonClient(keyGenerator.getDbName()).getScoredSortedSet(keyGenerator.gen());
    }


    @Override
    public Boolean add(KeyGenerator generator, Object value, double score) {
        if (value instanceof String) {
            return this.scoredSortedSetOperations(generator).add(score, value.toString());
        } else {
            return this.scoredSortedSetOperations(generator).add(score, this.serializerString(value));
        }
    }

    @Override
    public Long add(KeyGenerator generator, Set<TypedTuple> tuples) {

        return (long) this.scoredSortedSetOperations(generator).addAll(this.serialize(tuples));
    }

    private Map<String, Double> serialize(Set<TypedTuple> tuples) {
        Map<String, Double> map = new HashMap((int) (tuples.size() / 0.75f) + 1);
        for (TypedTuple next : tuples) {
            if (next.getValue() instanceof String) {
                map.put(next.getValue().toString(), next.getScore());
            } else {
                map.put(JSON.toJSONString(next.getValue()), next.getScore());
            }
        }
        return map;
    }

    @Override
    public <T> Double incrementScore(KeyGenerator generator, T value, double amount) {
        if (value instanceof String) {
            return this.scoredSortedSetOperations(generator).addScore(value.toString(), amount);
        } else {
            return this.scoredSortedSetOperations(generator).addScore(this.serializerString(value), amount);
        }
    }

    @Override
    public <T> Long remove(KeyGenerator generator, Object... values) {
        AtomicLong atomicLong = new AtomicLong();
        for (Object value : values) {
            boolean remove = this.scoredSortedSetOperations(generator).remove(this.serializerString(value));
            atomicLong.addAndGet(remove ? 1 : 0);
        }
        return atomicLong.get();
    }

    @Override
    public Long removeByRange(KeyGenerator generator, long start, long end) {
        return (long) this.scoredSortedSetOperations(generator).removeRangeByRank((int) start, (int) end);
    }

    @Override
    public Long removeByScore(KeyGenerator generator, double min, double max) {
        return (long) this.scoredSortedSetOperations(generator).removeRangeByScore(min, true, max, true);
    }

    @Override
    public Long size(KeyGenerator generator) {
        return (long) this.scoredSortedSetOperations(generator).size();
    }

    @Override
    public Long count(KeyGenerator generator, double min, double max) {
        return (long) this.scoredSortedSetOperations(generator).count(min, true, max, true);
    }

    @Override
    public <T> Long rank(KeyGenerator generator, T value) {
        return (long) this.scoredSortedSetOperations(generator).rank(this.serializerString(value));
    }

    @Override
    public <T> Long reverseRank(KeyGenerator generator, T value) {
        return (long) this.scoredSortedSetOperations(generator).revRank(this.serializerString(value));
    }

    @Override
    public <T> Double score(KeyGenerator generator, T value) {
        return this.scoredSortedSetOperations(generator).getScore(this.serializerString(value));
    }

    @Override
    public <T> Set<TypedTuple<T>> rangeByScoreWithScores(KeyGenerator generator, double min, double max,
                                                         long offset, long count, boolean isReversed, Class<T> clasz) {
        Collection<ScoredEntry<T>> result;
        if (isReversed) {
            result = this.scoredSortedSetOperations(generator).entryRangeReversed(min, true, max, true, (int) offset, (int) count);
        } else {
            result = this.scoredSortedSetOperations(generator).entryRange(min, true, max, true, (int) offset, (int) count);
        }
        return this.deserializeTuple(Objects.requireNonNull(result), clasz);
    }

    private <T> Set<TypedTuple<T>> deserializeTuple(Collection<ScoredEntry<T>> tuples, Class<T> clasz) {
        Set<TypedTuple<T>> result = new LinkedHashSet<>();
        if (null == tuples) {
            return result;
        }
        for (ScoredEntry<T> tuple : tuples) {
            result.add(new TypedTuple(this.deSerializerString(tuple.getValue().toString(), clasz, null), tuple.getScore()));
        }
        return result;
    }

    @Override
    public <T> Set<TypedTuple<T>> rangeByScoreWithScores(KeyGenerator generator, double min, double max,
                                                         boolean isReversed, Class<T> clasz) {
        Collection<ScoredEntry<T>> result;
        if (isReversed) {
            result = this.scoredSortedSetOperations(generator).entryRangeReversed(min, true, max, true);
        } else {
            result = this.scoredSortedSetOperations(generator).entryRange(min, true, max, true);
        }

        return this.deserializeTuple(Objects.requireNonNull(result), clasz);
    }


    @Override
    public <T> Set<T> range(KeyGenerator generator, long start, long end, boolean isReversed, Class<T> clasz) {
        Collection<String> result = new ArrayList<>();
        if (isReversed) {
            result = this.scoredSortedSetOperations(generator).valueRangeReversed((int) start, (int) end);
        } else {
            result = this.scoredSortedSetOperations(generator).valueRange((int) start, (int) end);
        }
        return listToSet(result, clasz);
    }

    private <T> Set<T> listToSet(Collection<String> result, Class<T> clasz) {
        Set<T> set = new LinkedHashSet<>();
        for (String str : result) {
            set.add(this.deSerializerString(str, clasz, null));
        }
        return set;
    }

    @Override
    public <T> Set<TypedTuple<T>> rangeWithScores(KeyGenerator generator, long start, long end, Class<T> clasz) {
        Collection<ScoredEntry<T>> result = this.scoredSortedSetOperations(generator).entryRange((int) start, (int) end);
        return deserializeTuple(result, clasz);
    }

    @Override
    public <T> Set<T> rangeByScore(KeyGenerator generator, double min, double max, boolean isReversed, Class<T> clasz) {
        Collection<String> result;
        if (isReversed) {
            result = this.scoredSortedSetOperations(generator).valueRangeReversed(min, true, max, true);
        } else {
            result = this.scoredSortedSetOperations(generator).valueRange(min, true, max, true);
        }
        return stringToSet(result, clasz);
    }

    private <T> Set<T> stringToSet(Collection<String> result, Class<T> clasz) {
        Set<T> set = new LinkedHashSet<>();
        for (String str : result) {
            set.add(this.deSerializerString(str, clasz, null));
        }
        return set;
    }

    private <T> Set<T> scoredEntryToSet(Collection<ScoredEntry<T>> result, Class<T> clasz) {
        Set<T> set = new LinkedHashSet<>();
        for (ScoredEntry<T> tScoredEntry : result) {
            set.add(this.deSerializerString(tScoredEntry.getValue().toString(), clasz, null));
        }
        return set;
    }

    @Override
    public <T> Set<T> rangeByScore(KeyGenerator generator, double min, double max, long offset,
                                   long count, boolean isReversed, Class<T> clasz) {
        Collection<String> result;
        if (isReversed) {
            result = this.scoredSortedSetOperations(generator).valueRangeReversed(min, true, max, true, (int) offset, (int) count);
        } else {
            result = this.scoredSortedSetOperations(generator).valueRange(min, true, max, true, (int) offset, (int) count);
        }
        return stringToSet(result, clasz);
    }

    @Override
    public Long intersectAndStore(KeyGenerator target, KeyGenerator source, KeyGenerator... otherKeys) {
        String[] strings = convertKeys(source, otherKeys).stream().toArray(String[]::new);
        return (long) this.scoredSortedSetOperations(target).intersection(strings);
    }


    @Override
    public Long unionAndStore(KeyGenerator target, KeyGenerator source, KeyGenerator... otherKeys) {
        return (long) this.scoredSortedSetOperations(target).union(convertKeys(source, otherKeys).stream().toArray(String[]::new));
    }

}
